#!/bin/bash

# 定义帮助信息
usage() {
    echo "Usage:"
    echo "  sudo dh_bond.sh set bond=<bond_name> mode=<1|4> eth1=<eth_name1> eth2=<eth_name2>"
    echo "  sudo dh_bond.sh set bond=<bond_name> mode=<1|4> eth1=<eth_name1> eth2=<eth_name2> [lacp_rate=<fast|slow>] [xmit_hash_policy=<hash_policy>] [fail_over_mac=<1|0>]"
    echo "  sudo dh_bond.sh del bond=<bond_name> eth1=<eth_name1> eth2=<eth_name2>"
    echo "Example:"
    echo "  sudo dh_bond.sh set bond=bond2 mode=1 eth1=ens5f0 eth2=ens5f1"
    echo "  sudo dh_bond.sh set bond=bond2 mode=4 eth1=ens5f0 eth2=ens5f1 lacp_rate=slow xmit_hash_policy=layer3+4"
    echo "Restrict:"
    echo "  eth1 is used as primary pf and can be used to create vf"
    echo "  mode 1 <-> active-backup | 4 <-> 802.3ad"
    echo "  The parameters <fail_over_mac> can only be used with mode 1"
    echo "  The parameters <lacp_rate> and <xmit_hash_policy> can only be used with mode 4"
    exit 1
}

# Check if the parameter is passed in
if [ "$#" -lt 1 ]; then
    usage
fi

# Parameter parsing
COMMAND=$1
shift

# Default Parameters
BOND_NAME=""
BOND_MODE=""
LACP_RATE=""
XMIT_HASH_POLICY=""
FAIL_OVER_MAC=""
ETH1=""
ETH2=""
CURRENT_DIR=$(dirname "$(readlink -f "$0")")

# Parsing key-value pair parameters
for ARG in "$@"; do
    case $ARG in
        bond=*)
            BOND_NAME="${ARG#*=}"
            ;;
        mode=*)
            BOND_MODE="${ARG#*=}"
            ;;
        lacp_rate=*)
            LACP_RATE="${ARG#*=}"
            ;;
        xmit_hash_policy=*)
            XMIT_HASH_POLICY="${ARG#*=}"
            ;;
        fail_over_mac=*)
            FAIL_OVER_MAC="${ARG#*=}"
            ;;
        eth1=*)
            ETH1="${ARG#*=}"
            ;;
        eth2=*)
            ETH2="${ARG#*=}"
            ;;
        *)
            echo "Invalid argument: $ARG"
            usage
            ;;
    esac
done

# Check if required parameters exist
if [[ "$COMMAND" == "set" ]]; then
    if [ -z "$BOND_NAME" ] || [ -z "$BOND_MODE" ] || [ -z "$ETH1" ] || [ -z "$ETH2" ]; then
        echo "Missing required parameters for 'set'."
        usage
    fi
elif [[ "$COMMAND" == "del" ]]; then
    if [ -z "$BOND_NAME" ] || [ -z "$ETH1" ] || [ -z "$ETH2" ]; then
        echo "Missing required parameters for 'del'."
        usage
    fi
else
    echo "Invalid command: $COMMAND"
    usage
fi

# Check bond netdev and their slaves
check_bond() {
    local bond=$1
    local eth1=$2
    local eth2=$3

    # Check if the bond exists
    if ! ip link show "$bond" &>/dev/null; then
        echo "Bond interface $bond does not exist."
        return 1
    fi

    # Get a list of slave devices on a bond
    local slaves
    slaves=$(cat /sys/class/net/"$bond"/bonding/slaves 2>/dev/null)

    if [ -z "$slaves" ]; then
        echo "Bond interface $bond exists but has no slave devices."
        return 2
    fi

    # Check if the slave devices are eth1 and eth2
    if [[ "$slaves" != *"$eth1"* || "$slaves" != *"$eth2"* ]]; then
        echo "Bond interface $bond exists with slaves: $slaves, but not matching $eth1 and $eth2."
        return 3
    fi

    echo "Bond interface $bond exists and correctly includes $eth1 and $eth2."
    return 0
}

# Function to check whether the network port exists
check_interface() {
    local eth_name1=$1
    local eth_name2=$2
    local ret=0

    if ! ip link show "$eth_name1" &>/dev/null; then
        echo "Error: Network interface $eth_name1 does not exist."
        ret=1
    fi

    if ! ip link show "$eth_name2" &>/dev/null; then
        echo "Error: Network interface $eth_name2 does not exist."
        ret=1
    fi

    return $ret
}

# Delete VF function
check_vf() {
    local eth_name1=$1
    local eth_name2=$2
    local ret=0

    vfs1=$(ip link show "$eth_name1" | grep -oE 'vf [0-9]+')
    vfs2=$(ip link show "$eth_name2" | grep -oE 'vf [0-9]+')

    if [ -n "$vfs1" ]; then
        echo "PF $eth_name1 has VFs, please remove VF and try again."
        ret=1
    fi
    if [ -n "$vfs2" ]; then
        echo "PF $eth_name2 has VFs, please remove VF and try again."
        ret=1
    fi
    return $ret
}

check_and_restart_network_manager() {
    # Check if the NetworkManager service exists
    if systemctl list-units --type=service | grep -qw "NetworkManager.service"; then
        echo "NetworkManager service exists."
        # Check if NetworkManager is running
        if [ "$(systemctl is-active NetworkManager)" = "active" ]; then
            echo "NetworkManager is running. Restart NetworkManager..."
            systemctl restart NetworkManager
            if [ $? -eq 0 ]; then
                echo "NetworkManager restart successfully."
                return 1
            else
                echo "Failed to restart NetworkManager."
                return 1
            fi
        else
            echo "NetworkManager is not running."
            return 2
        fi
    else
        echo "NetworkManager service does not exist."
        return 2
    fi
}

# Execute corresponding operations according to the command
case $COMMAND in
    set)
        # Setting default parameters
        if [ "$BOND_MODE" == "1" ]; then
            FAIL_OVER_MAC=${FAIL_OVER_MAC:-"1"}
        elif [ "$BOND_MODE" == "4" ]; then
            LACP_RATE=${LACP_RATE:-"slow"}
            XMIT_HASH_POLICY=${XMIT_HASH_POLICY:-"layer3+4"}
        else
            echo "Only support bond mode 1 (Active-Backup) or mode 4 (802.3ad)."
            exit 1
        fi

        # Load the bonding module
        modprobe bonding

        # Check bond and slave devices
        check_bond "$BOND_NAME" "$ETH1" "$ETH2"
        case $? in
            0)  # The bond network port already exists and the slave devices are eth1 and eth2
                exit 0
                ;;
            1|2) # The bond network port does not exist, or exists but has no slave device.
                ;;
            3)  # The bond network port exists but the slave device is not eth1 or eth2
                exit 1
                ;;
        esac

        # Check whether the slave network port exists
        check_interface "$ETH1" "$ETH2"
        if [ $? -ne 0 ]; then
            exit 1
        fi

        # Check VF device
        check_vf "$ETH1" "$ETH2"
        if [ $? -ne 0 ]; then
            exit 1
        fi

        # NIC hardware bond configuration
        echo "Set HardWare-Bond configuration for $ETH1 and $ETH2..."
        ethtool --set-priv-flags $ETH1 hardware-bond on
        ethtool --set-priv-flags $ETH2 hardware-bond on
        ethtool --set-priv-flags $ETH1 hardware-bond-primary on
        ethtool --set-priv-flags $ETH2 hardware-bond-primary off

        check_and_restart_network_manager
        ret_val=$?
        echo "Configuring bond: $BOND_NAME with mode: $BOND_MODE, lacp_rate: $LACP_RATE, xmit_hash_policy: $XMIT_HASH_POLICY, fail_over_mac: $FAIL_OVER_MAC on $ETH1 and $ETH2"
        sleep 1
        sudo python3 $CURRENT_DIR/dh_linux_bond.py set BOND="$BOND_NAME" MODE="$BOND_MODE" lacp_rate="$LACP_RATE" xmit_hash_policy="$XMIT_HASH_POLICY" fail_over_mac="$FAIL_OVER_MAC" eth1="$ETH1" eth2="$ETH2" network_manager_restart="$ret_val"
        if [ $ret_val -eq 1 ]; then
            sleep 2
            systemctl restart NetworkManager
            echo "NetworkManager restart successfully."
        fi

        ;;
    del)
        # Check bond and slave devices
        check_bond "$BOND_NAME" "$ETH1" "$ETH2"
        case $? in
            1)   # The bond network port does not exist
                exit 1
                ;;
            2|0) # The bond network port exists and the slave devices are eth1 and eth2, or it exists but has no slave devices.
                ;;
            3)   # The bond network port exists but the slave device is not eth1 or eth2
                exit 1
                ;;
        esac

        # Check whether the slave network port exists
        check_interface "$ETH1" "$ETH2"
        if [ $? -ne 0 ]; then
            exit 1
        fi

        # Check VF device
        check_vf "$ETH1" "$ETH2"
        if [ $? -ne 0 ]; then
            exit 1
        fi

        # Delete bond
        check_and_restart_network_manager
        ret_val=$?
        echo "Removing bond: $BOND_NAME on $ETH1 and $ETH2"
        sleep 1
        sudo python3 $CURRENT_DIR/dh_linux_bond.py del BOND="$BOND_NAME" eth1="$ETH1" eth2="$ETH2" network_manager_restart="$ret_val"

        # Remove HardWare-Bond configuration
        echo "Remove HardWare-Bond configuration for $ETH1 and $ETH2..."
        ethtool --set-priv-flags $ETH1 hardware-bond off
        ethtool --set-priv-flags $ETH2 hardware-bond off
        ethtool --set-priv-flags $ETH1 hardware-bond-primary on
        ethtool --set-priv-flags $ETH2 hardware-bond-primary on
        if [ $ret_val -eq 1 ]; then
            sleep 2
            systemctl restart NetworkManager
            echo "NetworkManager restart successfully."
        fi
        ;;
    *)
        echo "Unknown command: $COMMAND"
        usage
        ;;
esac
