import datetime
import os
import sys
import time

PATH_LOG = '/var/log/dhbond'
FILE_LOG = os.path.join(PATH_LOG, 'bond_config.log')
if not os.path.exists(PATH_LOG):
    os.makedirs(PATH_LOG)

PATH_NET_SCRIPT = '/etc/sysconfig/network-scripts/'
PATH_NET_SCRIPT_NIC_CFG = PATH_NET_SCRIPT + 'ifcfg-%s'
PATH_BOND_CONF = '/etc/modprobe.d/bonding.conf'


def do_log(data):
    time_local = str(datetime.datetime.now())
    with open(FILE_LOG, 'a') as log_file:
        log_file.write(f"{time_local}: {data}\n")

def cmd_exec(cmd):
    do_log("[cmd]: %s" % cmd)
    with os.popen(cmd) as f:
        rst = f.read().strip()
    if rst:
        do_log("[rst]: %s" % rst)
    return rst


def create_linux_bond(bond, mode, lacp_rate=None, xmit_hash_policy=None, fail_over_mac=None, slave_nic_list=[]):
    # Setting Default Values
    if mode == '1':
        fail_over_mac = fail_over_mac or '1'
    elif mode == '4':
        lacp_rate = lacp_rate or 'slow'
        xmit_hash_policy = xmit_hash_policy or 'layer3+4'

    # 1、Create a BOND interface configuration file
    path_ifcfg_bond = PATH_NET_SCRIPT_NIC_CFG % bond
    if not os.path.exists(PATH_BOND_CONF):
        cmd_exec("sudo touch %s" % PATH_BOND_CONF)

    cmd_exec("sudo touch %s" % path_ifcfg_bond)
    cmd_exec("sudo echo 'DEVICE=%s' >> %s" % (bond, path_ifcfg_bond))
    cmd_exec("sudo echo 'NAME=%s' >> %s" % (bond, path_ifcfg_bond))
    cmd_exec("sudo echo 'BOOTPROTO=static' >> %s" % path_ifcfg_bond)
    cmd_exec("sudo echo 'BONDING_MASTER=yes' >> %s" % path_ifcfg_bond)
    cmd_exec("sudo echo 'TYPE=Bond' >> %s" % path_ifcfg_bond)
    cmd_exec("sudo echo 'ONBOOT=yes' >> %s" % path_ifcfg_bond)
    cmd_exec("sudo echo 'RELOAD=no' >> %s" % path_ifcfg_bond)

    # Update BOND interface configuration file
    bonding_opts = f"miimon=100 mode={mode}"
    if mode == '1' and fail_over_mac == '1':
        bonding_opts += " fail_over_mac=1"
    if mode == '4':
        bonding_opts += f" xmit_hash_policy={xmit_hash_policy}"
        bonding_opts += f" lacp_rate={lacp_rate}"
    cmd_exec("sudo echo 'BONDING_OPTS=\"%s\"' >> %s" % (bonding_opts, path_ifcfg_bond))

    # 2、Update slave interface configuration file
    for nic in slave_nic_list:
        path_ifcfg_nic = PATH_NET_SCRIPT_NIC_CFG % nic
        if not os.path.exists(path_ifcfg_nic):
            cmd_exec("sudo touch %s" % path_ifcfg_nic)
            cmd_exec("sudo echo 'ONBOOT=yes' >> %s" % path_ifcfg_nic)
        cmd_exec("sudo sed -i '/DEVICE=/d' %s" % path_ifcfg_nic)
        cmd_exec("sudo echo 'DEVICE=%s' >> %s" % (nic, path_ifcfg_nic))
        cmd_exec("sudo sed -i '/NAME=/d' %s" % path_ifcfg_nic)
        cmd_exec("sudo echo 'NAME=%s' >> %s" % (nic, path_ifcfg_nic))
        cmd_exec("sudo sed -i '/MASTER=/d' %s" % path_ifcfg_nic)
        cmd_exec("sudo echo 'MASTER=%s' >> %s" % (bond, path_ifcfg_nic))
        cmd_exec("sudo sed -i '/SLAVE=/d' %s" % path_ifcfg_nic)
        cmd_exec("sudo echo 'SLAVE=yes' >> %s" % path_ifcfg_nic)
        cmd_exec("sudo sed -i '/RELOAD=/d' %s" % path_ifcfg_nic)
        cmd_exec("sudo echo 'RELOAD=no' >> %s" % path_ifcfg_nic)
        cmd_exec("sudo sed -i '/ONBOOT=/d' %s" % path_ifcfg_nic)
        cmd_exec("sudo echo 'ONBOOT=yes' >> %s" % path_ifcfg_nic)

    # 3、Update /etc/modprobe.d/bonding.conf
    cmd_exec("sudo echo 'alias %s bonding' >>%s" % (bond, PATH_BOND_CONF))
    cmd_exec("sudo echo 'options %s %s' >>%s" % (bond, bonding_opts, PATH_BOND_CONF))

    # 4、Update BOND parameters in the kernel
    cmd_exec("sudo echo '+%s' > /sys/class/net/bonding_masters" % bond)
    cmd_exec("sudo echo '%s' > /sys/class/net/%s/bonding/mode" % (mode, bond))
    if mode == '4':
        cmd_exec("sudo echo '%s' > /sys/class/net/%s/bonding/xmit_hash_policy" % (xmit_hash_policy, bond))
        cmd_exec("sudo echo '%s' > /sys/class/net/%s/bonding/lacp_rate" % (lacp_rate, bond))
    if mode == '1' and fail_over_mac == '1':
        cmd_exec("sudo echo 1 > /sys/class/net/%s/bonding/fail_over_mac" % bond)
    for nic in slave_nic_list:
        cmd_exec("sudo ip link set %s down" % nic)
        cmd_exec("sudo echo '+%s' > /sys/class/net/%s/bonding/slaves" % (nic, bond))

    # 5、Activate BOND
    cmd_exec("sudo ip link set %s up" % bond)

def delete_linux_bond(bond, slave_nic_list, network_manager_restart=None):
    # 1. Deleting the BOND interface configuration file
    path_ifcfg_bond = PATH_NET_SCRIPT_NIC_CFG % bond
    cmd_exec(f"sudo rm -rf {path_ifcfg_bond}")

    # 2. Clean up the BOND settings in the slave NIC configuration files
    for nic in slave_nic_list:
        path_ifcfg_nic = PATH_NET_SCRIPT_NIC_CFG % nic
        if os.path.exists(path_ifcfg_nic):
            cmd_exec(f"sudo sed -i '/MASTER=/d' {path_ifcfg_nic}")
            cmd_exec(f"sudo sed -i '/SLAVE=/d' {path_ifcfg_nic}")
            cmd_exec(f"sudo sed -i '/RELOAD=/d' {path_ifcfg_nic}")

    if network_manager_restart == '1':
        cmd_exec("sudo systemctl restart NetworkManager")
        time.sleep(2)

    # 3. Removing a slave device from a BOND interface
    slave_nic_str = " ".join(slave_nic_list)
    cmd_exec(f"sudo ifenslave -d {bond} {slave_nic_str}")

    # 4. Clear /etc/modprobe.d/bonding.conf
    cmd_exec(f"sudo sed -i '/alias {bond} bonding/d' {PATH_BOND_CONF}")
    cmd_exec(f"sudo sed -i '/options {bond} /d' {PATH_BOND_CONF}")

    # 5. Remove BOND settings from the kernel
    cmd_exec(f"sudo echo '-{bond}' > /sys/class/net/bonding_masters")

    # 6. Activate the Eth interface
    for nic in slave_nic_list:
        cmd_exec(f"sudo ip link set {nic} nomaster")
        cmd_exec(f"sudo ip link set dev {nic} up")


def do_proc(argv):
    do_log("start proc")
    if argv[1] == 'set':
        args = dict(arg.split('=') for arg in argv[2:])
        bond = args.get('BOND')
        mode = args.get('MODE')
        lacp_rate = args.get('lacp_rate')
        xmit_hash_policy = args.get('xmit_hash_policy')
        fail_over_mac = args.get('fail_over_mac')
        slave_nic_list = [args.get('eth1'), args.get('eth2')]
        create_linux_bond(bond, mode, lacp_rate, xmit_hash_policy, fail_over_mac, slave_nic_list)
    elif argv[1] == 'del':
        args = dict(arg.split('=') for arg in argv[2:])
        bond = args.get('BOND')
        slave_nic_list = [args.get('eth1'), args.get('eth2')]
        network_manager_restart = args.get('network_manager_restart')
        delete_linux_bond(bond, slave_nic_list, network_manager_restart)
    else:
        raise ValueError("Invalid command")
    do_log("end proc")


if __name__ == "__main__":
    do_proc(sys.argv)
