#!/bin/bash

set +e

print_help() {
    echo "Usage: $0 {--<operations> <drv1> <drv2>...} [--ofed] [--ofed_path <path>] [--ofed_version <version>] [--abs_ofed_path]"
    echo "operations:       compile, install, uninstall"
    echo "Available drvs:   eth, rdma(need eth), rdma-core, all"
    echo "ofed:             compile using OFED"
    echo "ofed_path:        ofed real path"
    echo "ofed_version:     ofed version"
    echo "abs_ofed_path:    ofed absolute path"
}

# 检查是否传入了操作类型
if [ $# -lt 2 ]; then
    print_help
    exit 1
fi

ROOTDIR="$( cd "$( dirname "$0" )" && pwd )"

declare -A drivers_path
drivers_path["eth"]="$ROOTDIR/zte/zxdh_kernel/kernel-opensrc"   # eth 驱动的压缩包路径
drivers_path["sda"]="$ROOTDIR/zxdh-sda-*" # sda 驱动的压缩包路径
drivers_path["rdma"]="$ROOTDIR/zxdh-rdma-*/zrdma" # rdma 驱动的压缩包路径
drivers_path["rdma-core"]="$ROOTDIR/rdma-core*" # rdma 驱动的压缩包路径

declare -A drivers_tgz
drivers_tgz["eth"]="zxdh-eth-*.src.tgz"
drivers_tgz["rdma"]="zxdh-rdma-*.src.tgz"
drivers_tgz["sda"]="zxdh-sda-*.src.tgz"
drivers_tgz["rdma-core"]="zxdh-rdma-core-*.src.tgz"

params=""
# ofed_version=""
# ofed_path=""
# ofed=""
# use_abs_path=""

while [ "$#" -gt 0 ]; do
  case "$1" in
    --*) # 如果参数以 -- 开头，则是 operation
      current_operation=${1#--}
      shift
      # 检查是否有后续参数属于该 operation
      case "$current_operation" in
        "ofed")
            ofed=("$current_operation")
            ;;
        "ofed_version")
            ofed_version=$1
            shift
            ;;
        "ofed_path")
            ofed_path=$1
            shift
            ;;
        "abs_ofed_path")
            abs_ofed_path=("$current_operation")
            ;;
        "compile"|"install"|"uninstall")
            operation=("$current_operation")
            ;;
        "help")
            print_help
      esac
      ;;
    *) # 默认是驱动名称
      params+=("$1")
    #   echo "driver is $params $1"
      shift
      ;;
  esac
done

if [[ "$params" == "all" ]]; then
    drivers_tmp="eth rdma rdma-core"
else
    drivers_tmp=${params[@]}
fi

echo "$operation : $drivers_tmp"

if [[ "$drivers_tmp" =~ (^|[[:space:]])rdma($|[[:space:]]) ]] && ! [[ "$drivers_tmp" =~ (^|[[:space:]])eth($|[[:space:]]) ]]; then
    echo "Error: 'rdma' requires 'eth'."
    exit 1
fi

drivers=""
drivers_sup="eth rdma rdma-core sda"

for driver1 in $drivers_sup; do
    for driver2 in $drivers_tmp; do
        if [[ "$driver1" == "$driver2" ]]; then
            drivers="$drivers $driver1"
        fi
    done
done
drivers=$(echo "$drivers" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')

drivers_round=$(echo "$drivers" | awk '{for(i=NF; i>0; i--) printf "%s ", $i; print ""}')

for driver in $drivers; do
    if compgen -d ${drivers_path[$driver]} > /dev/null; then
        echo "$driver Path exists."
    else
        if compgen -G ${drivers_tgz[$driver]} > /dev/null; then
            echo "Decompress $driver"
            if [[ $driver == "rdma" ]]; then
                comp_file=$(ls zxdh-rdma-*.src.tgz 2>/dev/null | grep -v rdma-core)
                tar -xzvf "$comp_file" > /dev/null
            else
                tar -zxvf ${drivers_tgz[$driver]} > /dev/null
            fi
        else
            echo "$driver does not exist. ${drivers_tgz[$driver]}"
            exit 1
        fi
    fi
done

compile_rdma() {
    if [ -n "$ofed" ] && [ -z "$ofed_path" ] && [ -z "$ofed_version" ]; then
        echo "use /usr/src/ofa_kernel!!"
        eval cd "${drivers_path[$driver]}" && ./build.sh --use_abs_net_path ${drivers_path["eth"]} ofed > /dev/null
    elif [ -n "$ofed" ] && [ -z "$ofed_path" ] && [ -n "$ofed_version" ]; then
        echo "use /usr/src/ofa_kernel ofed_version $ofed_version!!"
        eval cd "${drivers_path[$driver]}" && ./build.sh --use_abs_net_path ${drivers_path["eth"]} ofed --ofed_version "$ofed_version"> /dev/null
    elif [ -n "$ofed" ] && [ -n "$ofed_path" ] && [ -z "$ofed_version" ]; then
        echo "use $ofed_path ofed_version!!"
        eval cd "${drivers_path[$driver]}" && ./build.sh --use_abs_net_path ${drivers_path["eth"]} --ofed_path "$ofed_path" > /dev/null
    elif [ -n "$ofed" ] && [ -n "$ofed_path" ] && [ -n "$ofed_version" ]; then
        echo "use $ofed_path ofed_version $ofed_version!!"
        eval cd "${drivers_path[$driver]}" && ./build.sh --use_abs_net_path ${drivers_path["eth"]} --ofed_path "$ofed_path" --ofed_version "$ofed_version" > /dev/null
    elif [ -z "$ofed" ] && [ -z "$abs_ofed_path" ]; then
        eval cd "${drivers_path[$driver]}" && ./build.sh --use_abs_net_path ${drivers_path["eth"]} > /dev/null
    elif [ -n "$abs_ofed_path" ]; then
         echo "use /usr/src/kernel/5.10.0-1.0.0.35/mlnx-ofa_kernel-5.8"
        eval cd "${drivers_path[$driver]}" && ./build.sh --use_abs_net_path ${drivers_path["eth"]} --ofed_version "5.8" --ofed_path "/usr/src/kernel/5.10.0-1.0.0.35/mlnx-ofa_kernel-5.8" > /dev/null
    fi
}

# 定义编译函数
compile_driver() {
    local driver=$1
    case "$driver" in
        sda)
            echo "Compiling sda..."
            eval cd "${drivers_path[$driver]}/" && make > /dev/null
            ;;
        eth)
            echo "Compiling eth..."
            eval cd "${drivers_path[$driver]}" && ./build.sh m > /dev/null
            ;;
        rdma)
            echo "Compiling rdma..."
            compile_rdma
            ;;
        rdma-core)
            echo "Compiling rdma-core..."
            eval cd "${drivers_path[$driver]}/" && mkdir -p build && cd build 
            cmake .. > /dev/null
            make -j7 > /dev/null
            ;;
        all)
            echo "Compiling all..."
            eval cd ${drivers_path["sda"]} && make > /dev/null
            eval cd ${drivers_path["eth"]} && ./build.sh m > /dev/null
            compile_rdma
            ;;
        *)
            echo "Unknown driver: $driver"
            return 1
            ;;
    esac
}

# 定义安装函数
install_driver() {
    local driver=$1
    case "$driver" in
        sda)
            echo "Installing sda..."
            eval cd "${drivers_path[$driver]}" 
            insmod ./accdevice/zsda_common/zsda_common.ko
            insmod ./accdevice/zsda_pf/zsda_pf.ko
            insmod ./accdevice/zsda_vf/zsda_vf.ko
            if [[ $? -ne 0 ]]; then
                echo "Error installing sda. Continuing with other drivers..."
            fi
            ;;
        eth)
            echo "Installing eth..."
            eval cd "${drivers_path[$driver]}" && ./build.sh i
            if [[ $? -ne 0 ]]; then
                echo "Error installing eth. Continuing with other drivers..."
            fi
            ;;
        rdma)
            echo "Installing rdma..."
            eval cd "${drivers_path[$driver]}"
            modprobe ib_uverbs && insmod src/zrdma.ko
            if [[ $? -ne 0 ]]; then
                echo "Error installing rdma. Continuing with other drivers..."
            fi
            ;;
        rdma-core)
            echo "Installing rdma-core..."
            eval cd "${drivers_path[$driver]}/build"
            make install > /dev/null
            ldconfig
            if ! ldconfig -p | grep -q rdma; then
                echo "未找到 rdma, 开始刷新缓存..."
                cd /etc/ld.so.conf.d || exit 1
                echo "/usr/local/lib64" | sudo tee local_x86_64.conf > /dev/null
                sudo ldconfig
            fi
            ;;
        *)
            echo "Unknown driver: $driver"
            return 1
            ;;
    esac
}

uninstall_driver() {
    local driver=$1
    case "$driver" in
        sda)
            rmmod zsda_vf
            rmmod zsda_pf
            rmmod zsda_common
            if [[ $? -ne 0 ]]; then
                echo "Error uninstalling rdma. Continuing with other drivers..."
            fi
            ;;
        eth)
            eval cd "${drivers_path[$driver]}" && ./build.sh r
            if [[ $? -ne 0 ]]; then
                echo "Error uninstalling eth. Continuing with other drivers..."
            fi
            ;;
        rdma)
            rmmod zrdma
            if [[ $? -ne 0 ]]; then
                echo "Error uninstalling rdma. Continuing with other drivers..."
            fi
            ;;
        rdma-core)
            ;;
        *)
            echo "Unknown driver: $driver"
            return 1
            ;;
    esac
}

# 执行操作
case "$operation" in
    compile)
        echo "Starting compilation..."
        for driver in $drivers; do
            compile_driver "$driver" || exit 1
        done
        echo "All specified drivers compiled successfully, All ko is:"
        for driver in $drivers; do
            find ${drivers_path[$driver]} -type f -name *.ko
        done
        ;;
    install)
        echo "Starting installation..."
        for driver in $drivers; do
            install_driver "$driver" || exit 1
        done
        echo "All specified drivers installed successfully!"
        ;;
    uninstall)
        echo "Starting uninstallation..."
        for driver in $drivers_round; do
            uninstall_driver "$driver" || exit 1
        done
        echo "All specified drivers uninstalled successfully!"
        ;;
    *)
        echo "Unknown operation: $operation"
        echo "Available operations: compile, install, uninstall"
        exit 1
        ;;
esac

set -e
