#ifndef _DPP_DTB_TABLE_API_H_
#define _DPP_DTB_TABLE_API_H_

#ifdef __cplusplus
extern "C"{
#endif

#include "dpp_dev.h"
#include "zxic_common.h"
#include "dpp_stat_api.h"

#define DPP_DTB_DUMP_ZCAM_TYPE  (ZXIC_UINT32)(0)
#define DPP_DTB_DUMP_DDR_TYPE   (ZXIC_UINT32)(1)

typedef struct dpp_dtb_user_entry_t
{
    ZXIC_UINT32 sdt_no;        /*流表sdt号*/
    ZXIC_VOID* p_entry_data;   /*该流表的数据，数据结构见各表的结构体定义*/
}DPP_DTB_USER_ENTRY_T;

typedef struct dpp_dtb_eram_entry_info_t
{
    ZXIC_UINT32 index;      /*条目index，根据wrt_mode模式为单位的的index，支持1/64/128bit */
    ZXIC_UINT32 *p_data;    /*写入的表项信息 */
}DPP_DTB_ERAM_ENTRY_INFO_T;

typedef struct dpp_dtb_ddr_entry_info_t
{
    ZXIC_UINT32 index;      /*条目index，根据wrt_mode模式定义的index，支持128/256/384/512bit*/
    ZXIC_UINT32 *p_data;    /*写入的表项信息*/
}DPP_DTB_DDR_ENTRY_INFO_T;

typedef struct dpp_dtb_hash_entry_info_t
{
    ZXIC_UINT8 *p_actu_key;     /*实际的键值，对于一种表来说键值长度是固定的，在初始化时，会通过dpp_hash_tbl_id_info_init进行配置*/
    ZXIC_UINT8 *p_rst;          /*hash表结果 result的长度由当前业务结果位宽决定*/
}DPP_DTB_HASH_ENTRY_INFO_T;

typedef struct dpp_dtb_acl_entry_info_t
{
    ZXIC_UINT32   handle;     /*条目索引*/
    ZXIC_UINT8    *key_data;  /*键值data部分 按mode的长度，支持640bit/320bit/160bit/80bit*/
    ZXIC_UINT8    *key_mask;  /*键值mask部分 长度与data相同*/
    ZXIC_UINT8    *p_as_rslt; /*关联结果，仅使能关联查找情况有效 支持1/64/128bit*/
}DPP_DTB_ACL_ENTRY_INFO_T;

typedef struct dpp_dtb_acl_stat_info_t
{
    ZXIC_UINT32   handle;     /*条目索引*/
    ZXIC_UINT8    *stat_value;      /*统计值 按dump mode的长度，支持64bit/128bit*/
}DPP_DTB_ACL_STAT_INFO_T;

typedef struct dpp_dtb_dump_index_t
{
    ZXIC_UINT32 index;      /*index*/
    ZXIC_UINT32 index_type;  /*index类型 */
}DPP_DTB_DUMP_INDEX_T;

/**dtb 队列 dump地址红黑树初始化*/
ZXIC_UINT32 dpp_dtb_dump_addr_rb_init(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id);

/**dtb 队列 dump地址红黑树删除*/
ZXIC_UINT32 dpp_dtb_dump_addr_rb_destroy(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id);

/*dump地址信息获取*/
ZXIC_UINT32 dpp_dtb_dump_sdt_addr_get(ZXIC_UINT32 dev_id,
                                      ZXIC_UINT32 queue_id,  
                                      ZXIC_UINT32 sdt_no, 
                                      ZXIC_UINT64 *phy_addr, 
                                      ZXIC_UINT64 *vir_addr, 
                                      ZXIC_UINT32 *size);

/***********************************************************/
/** DTB通道申请
* @param   devId       NP设备号
* @param   pName       申请DTB通道的唯一设备名(最大32字符)
* @param   pQueueId    申请到的DTB通道编号
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/
ZXIC_UINT32 dpp_dtb_queue_requst(ZXIC_UINT32 devId,
                                 ZXIC_CHAR pName[32], 
                                 ZXIC_UINT16 vPort, 
                                 ZXIC_UINT32 *pQueueId);

/***********************************************************/
/** DTB通道释放
* @param   devId       NP设备号
* @param   pName       要释放DTB通道的唯一设备名(最大32字符)
* @param   pQueueId    要释放的DTB通道编号
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/
ZXIC_UINT32 dpp_dtb_queue_release(ZXIC_UINT32 devId,
                                  ZXIC_CHAR pName[32],  
                                  ZXIC_UINT32 queueId);

/***********************************************************/
/** DTB通道用户信息配置
* @param   devId      NP设备号
* @param   queueId    DTB通道编号
* @param   vPort      vport信息
* @param   vector     中断号
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/
ZXIC_UINT32 dpp_dtb_user_info_set(ZXIC_UINT32 devId,
                                  ZXIC_UINT32 queueId, 
                                  ZXIC_UINT16 vPort, 
                                  ZXIC_UINT32 vector);

/***********************************************************/
/** DTB通道下表空间地址设置,空间大小[32*(16+16*1024)B]
* @param   devId      NP设备号
* @param   queueId    DTB通道编号
* @param   phyAddr    物理地址
* @param   virAddr    虚拟地址
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/
ZXIC_UINT32 dpp_dtb_queue_down_table_addr_set(ZXIC_UINT32 devId,
                                              ZXIC_UINT32 queueId, 
                                              ZXIC_UINT64 phyAddr, 
                                              ZXIC_UINT64 virAddr);

/***********************************************************/
/** DTB通道dump空间地址设置,空间大小[32*(16+16*1024)B]
* @param   devId      NP设备号
* @param   pName      要释放DTB通道的设备名
* @param   phyAddr    物理地址
* @param   virAddr    虚拟地址
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/
ZXIC_UINT32 dpp_dtb_queue_dump_table_addr_set(ZXIC_UINT32 devId,
                                              ZXIC_UINT32 queueId, 
                                              ZXIC_UINT64 phyAddr, 
                                              ZXIC_UINT64 virAddr);

/***********************************************************/
/** 大批量dump一个流表使用的地址空间配置
* @param   devId       NP设备号
* @param   queueId     DTB队列编号
* @param   sdtNo       流表std号
* @param   phyAddr     物理地址
* @param   virAddr     虚拟地址
* @param   size        (最大64MB)
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/
ZXIC_UINT32 dpp_dtb_dump_sdt_addr_set(ZXIC_UINT32 devId,
                                      ZXIC_UINT32 queueId,  
                                      ZXIC_UINT32 sdtNo, 
                                      ZXIC_UINT64 phyAddr, 
                                      ZXIC_UINT64 virAddr, 
                                      ZXIC_UINT32 size);

/***********************************************************/
/** 清除大批量dump一个流表使用的地址空间配置
* @param   devId        NP设备号
* @param   queueId      DTB队列编号
* @param   sdtNo        流表std号
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/
ZXIC_UINT32 dpp_dtb_dump_sdt_addr_clear(ZXIC_UINT32 devId,
                                        ZXIC_UINT32 queueId, 
                                        ZXIC_UINT32 sdtNo);


/*获取批量流总数统计*/
ZXIC_UINT32 dpp_dtb_multi_flow_statistic_get(ZXIC_UINT32 devId);

/*获取批量流总数统计*/
ZXIC_UINT32 dpp_dtb_multi_write_reg_num_get(ZXIC_UINT32 devId);

/*获取批量流下表执行次数统计*/
ZXIC_UINT32 dpp_dtb_multi_down_process_num_get(ZXIC_UINT32 devId);

/*获取批量流定时执行次数统计*/
ZXIC_UINT32 dpp_dtb_multi_time_process_num_get(ZXIC_UINT32 devId);

/*批量流下表门限设置*/                                       
ZXIC_UINT32 dpp_dtb_multi_flow_threshold_set(ZXIC_UINT32 devId, ZXIC_UINT32 threshold);

/*获取批量流下表门限*/
ZXIC_UINT32 dpp_dtb_multi_flow_threshold_get(ZXIC_UINT32 devId);

/*批量流配置初始化*/
ZXIC_UINT32 dpp_dtb_multi_entries_init(ZXIC_UINT32 devId);

/*批量流配置销毁*/
ZXIC_UINT32 dpp_dtb_multi_entries_destroy(ZXIC_UINT32 devId);

/***********************************************************/
/** DTB 批量流下表超时回调
* *@param   devId       NP设备号
* @param   queueId     DTB队列编号
* @return
* @remark  无
* @see
* @author  cbb      @date  2024/04/08
************************************************************/
ZXIC_UINT32 dpp_dtb_multi_entries_timeout_handle(ZXIC_UINT32 devId, ZXIC_UINT32 queueId);

/***********************************************************/
/** DTB 批量流下表 适用eram、ddr、hash、ACL表
* @param   devId       NP设备号
* @param   queueId     DTB队列编号
* @param   entryNum    下表条目数
* @param   pDownEntrys 下表条目数据，数据内容格式参照上报各个表结构体定义
* @return
* @remark  无
* @see
* @author  cbb      @date  2024/04/08
************************************************************/
ZXIC_UINT32 dpp_dtb_multi_entries_write(ZXIC_UINT32 devId,
                                       ZXIC_UINT32 queueId,
                                       ZXIC_UINT32 entryNum,
                                       DPP_DTB_USER_ENTRY_T *pDownEntrys);

/***********************************************************/
/** DTB流表表项下表 适用eram、ddr、hash、ACL表
* @param   devId       NP设备号
* @param   queueId     DTB队列编号
* @param   entryNum    下表条目数
* @param   pDownEntrys 下表条目数据，数据内容格式参照上报各个表结构体定义
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/
ZXIC_UINT32 dpp_dtb_table_entry_write(ZXIC_UINT32 devId,
                                       ZXIC_UINT32 queueId,
                                       ZXIC_UINT32 entryNum,
                                       DPP_DTB_USER_ENTRY_T *pDownEntrys);

/***********************************************************/
/** DTB流表表项大批量dump 适用eram、ddr、hash、ACL表
* @param   devId           NP设备号
* @param   queueId         DTB队列编号
* @param   sdtNo           流表std号
* @param   startIndex      本次dump开始index
* @param   pDumpData       本次dump出的有效数据，数据格式与下表格式相同
* @param   entryNum        本次dump实际的有效条目数
* @param   nextStartIndex  下次dump是开始的index  
* @param   finishFlag      整个表dump完成标志，1表示完成，0表示未完成
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/   
 DPP_STATUS dpp_dtb_table_dump(ZXIC_UINT32 devId,
                               ZXIC_UINT32 queueId,
                               ZXIC_UINT32 sdtNo,
                               DPP_DTB_DUMP_INDEX_T startIndex,
                               ZXIC_UINT8* pDumpData,
                               ZXIC_UINT32 *entryNum,
                               DPP_DTB_DUMP_INDEX_T *nextStartIndex,
                               ZXIC_UINT32 *finishFlag);                            

/***********************************************************/
/** DTB整个流表清空Flush 适用于eram、ddr、ACL表
* @param   devId      NP设备号
* @param   queueId    DTB队列编号
* @param   sdtNo      流表std号
* @param   flush_mode flush模式 DPP_DDR_FLUSH_MODE_E 0：寄存器方式 1：dtb方式，默认选择寄存器方式，参数仅对hash表/ddr直接表有效
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/  
DPP_STATUS dpp_dtb_table_flush(ZXIC_UINT32 devId,
                               ZXIC_UINT32 queueId,
                               ZXIC_UINT32 sdtNo,
                               ZXIC_UINT32 flush_mode);

/***********************************************************/
/** DTB流表表项删除 适用eram、ddr、hash、ACL表
* @param   devId      NP设备号
* @param   queueId    DTB队列编号
* @param   entryNum   下表条目数
* @param   pDeleteEntrys 下表条目数据，数据内容格式参照上报各个表结构体定义
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/  
DPP_STATUS dpp_dtb_table_entry_delete(ZXIC_UINT32 devId,
                                       ZXIC_UINT32 queueId,
                                       ZXIC_UINT32 entryNum,
                                       DPP_DTB_USER_ENTRY_T *pDeleteEntrys);

/***********************************************************/
/** DTB流表表项获取 适用eram、ddr、hash、ACL表
* @param   devId      NP设备号
* @param   queueId    DTB队列编号
* @param   pGetEntry  获取条目数据，数据内容格式参照上报各个表结构体定义
* @param   srh_mode   查找模式软件or硬件 DPP_HASH_SRH_MODE，仅对hash表有效
* @return
* @remark  无
* @see
* @author  cbb      @date  2023/07/03
************************************************************/ 
DPP_STATUS dpp_dtb_entry_get(ZXIC_UINT32 devId,
                             ZXIC_UINT32 queueId,
                             DPP_DTB_USER_ENTRY_T *pGetEntry,
                             ZXIC_UINT32 srh_mode);

/***********************************************************/
/** 释放所有se资源和流表(通过dtb方式)
* @param   dev_id    设备号 
* @param   queue_id  队列号 
* @param   flush_mode  ddr删除模式 0：寄存器模式 1：dtb模式，参考DPP_DDR_FLUSH_MODE_E
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/09/19
************************************************************/
DPP_STATUS dpp_dtb_res_destroy(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id, ZXIC_UINT32 flush_mode);

/***********************************************************/
/** 离线释放当前sdt下的所有片内和片外流表表项
* (适用于进程启动后，仅配置流表资源，软件未配置流表，但需要删除硬件上已配置流表的场景)
* @param   dev_id  设备号 
* @param   sdt_no  sdt号 
* @param   queue_id  队列号
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/12/04
************************************************************/
DPP_STATUS dpp_dtb_hash_offline_delete(ZXIC_UINT32 dev_id,
                                        ZXIC_UINT32 queue_id,
                                        ZXIC_UINT32 sdt_no,
                                        ZXIC_UINT32 flush_mode);

/***********************************************************/
/** 在线释放当前sdt下的所有流表表项
*  进程正常退出前删除表项，软件上有存储表项,遍历软件上存储的表项，再逐个删除;
*  适用于纯片内，zcell/zreg非独占场景或者混合模式插入表项较少的场景
* @param   dev_id  设备号 
* @param   queue_id  队列号
* @param   sdt_no  sdt号
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/12/04
************************************************************/
DPP_STATUS dpp_dtb_hash_online_delete(ZXIC_UINT32 dev_id,
                                        ZXIC_UINT32 queue_id,
                                        ZXIC_UINT32 sdt_no);

/***********************************************************/
/** 在线释放当前sdt下的所有片内和片外流表表项
* (适用于进程正常退出前删除表项，软件上有存储表项)
*  适用于混合模式，插入表项较多的场景
* @param   dev_id  设备号 
* @param   queue_id  队列号
* @param   sdt_no    sdt号
* @param   flush_mode  ddr删除模式 0：寄存器模式 1：dtb模式，参考DPP_DDR_FLUSH_MODE_E
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/12/04
************************************************************/
DPP_STATUS dpp_dtb_hash_online_delete_ex(ZXIC_UINT32 dev_id,
                                        ZXIC_UINT32 queue_id,
                                        ZXIC_UINT32 sdt_no,
                                        ZXIC_UINT32 flush_mode);

/***********************************************************/
/** ppu计数值获取(dtb通道方式)
* @param   dev_id           设备号
* @param   queue_id         队列号
* @param   rd_mode          读取位宽模式，参见STAT_CNT_MODE_E，0-64bit，1-128bit
* @param   index            索引，具体位宽参见rd_mode
* @param   p_data           出参，读取的数据
*
* @return  NPE_OK-成功，NPE_ERR-失败
* @remark  无
* @see
* @author  cq      @date  2024/02/01
************************************************************/
DPP_STATUS dpp_dtb_stat_ppu_cnt_get(ZXIC_UINT32 dev_id,
                                ZXIC_UINT32 queue_id,
                                STAT_CNT_MODE_E rd_mode,
                                ZXIC_UINT32 index,
                                ZXIC_UINT32 *p_data);

/***********************************************************/
/** DTB dump统计表项 适用eram、ddr(最大dump16KB的数据)
* @param   devId           NP设备号
* @param   queue_id        DTB队列编号
* @param   dump_mode       dump类型 0-64bit 1-128bit
* @param   startIndex      本次dump开始index（64/128bit为单位）
* @param   entryNum        本次dump条目数(64/128bit为单位)
* @param   pDumpData       本次dump出的有效数据，数据格式与下表格式相同
* @return
* @remark  无
* @see
* @author  zth      @date  2024/06/17
************************************************************/  
DPP_STATUS dpp_dtb_ppu_stat_data_dump(ZXIC_UINT32 dev_id,
                                      ZXIC_UINT32 queue_id,
                                      ZXIC_UINT32 dump_mode,
                                      ZXIC_UINT32 startIndex,
                                      ZXIC_UINT32 entryNum,
                                      ZXIC_UINT8* pDumpData);

/***********************************************************/
/** acl index资源申请
* @param   dev_id          NP设备号
* @param   sdt_no          流表sdt号(0~255)
* @param   vport           端口号
* @param   p_index         申请到的索引值，acl下表时使用
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_acl_index_request(ZXIC_UINT32 dev_id,
                                     ZXIC_UINT32 sdt_no,
                                     ZXIC_UINT32 vport,
                                     ZXIC_UINT32 *p_index);

/***********************************************************/
/** acl index资源释放
* @param   dev_id          NP设备号
* @param   sdt_no          流表sdt号(0~255)
* @param   vport           端口号
* @param   index           需要释放的索引值
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_acl_index_release(ZXIC_UINT32 dev_id,
                                     ZXIC_UINT32 sdt_no,
                                     ZXIC_UINT32 vport,
                                     ZXIC_UINT32 index);

/***********************************************************/
/** 离线删除与vport关联的acl表项和索引值
* @param   dev_id          NP设备号
* @param   queue_id        dtb通道队列号(0~127)
* @param   sdt_no          流表sdt号(0~255)
* @param   vport           端口号
* @param   counter_id      统计编号，对应微码中的address
* @param   rd_mode         统计读取方式 0:64bit 1:128bit
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_acl_offline_delete(ZXIC_UINT32 dev_id,
                                      ZXIC_UINT32 queue_id,
                                      ZXIC_UINT32 sdt_no,
                                      ZXIC_UINT32 vport,
                                      ZXIC_UINT32 counter_id,
                                      ZXIC_UINT32 rd_mode);

/***********************************************************/
/** dump出指定vport的acl表项
* @param   dev_id          NP设备号
* @param   queue_id        dtb通道队列号(0~127)
* @param   sdt_no          流表sdt号(0~255)
* @param   vport           端口号
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_acl_table_dump_by_vport(ZXIC_UINT32 dev_id,
                                           ZXIC_UINT32 queue_id,
                                           ZXIC_UINT32 sdt_no,
                                           ZXIC_UINT32 vport,
                                           ZXIC_UINT32* entry_num,
                                           ZXIC_UINT8* pDumpData);

/***********************************************************/
/** dump出指定vport的stat统计
* @param   dev_id          NP设备号
* @param   queue_id        dtb通道队列号(0~127)
* @param   sdt_no          流表sdt号(0~255)
* @param   vport           端口号
* @param   dump_mode       dump模式 0：64bit 1：128bit STAT_CNT_MODE_E
* @param   start_index     dump起始偏移地址
* @param   entry_num       出参，dump出的实际统计个数 
* @param   pDumpData       出参，结构体DPP_DTB_ACL_STAT_INFO_T
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_acl_stat_dump_by_vport(ZXIC_UINT32 dev_id,
                                               ZXIC_UINT32 queue_id,
                                               ZXIC_UINT32 sdt_no,
                                               ZXIC_UINT32 vport,
                                               STAT_CNT_MODE_E dump_mode,
                                               ZXIC_UINT32 start_index,
                                               ZXIC_UINT32* entry_num,
                                               ZXIC_UINT8* pDumpData);

/***********************************************************/
/** 统计计数读清(根据统计id区分片内统计或者片外统计，片内统计dtb清0，片外统计代理通道发送消息给固件清0，非代理通道读清)
* @param   dev_id           NP设备号
* @param   queue_id         队列号
* @param   rd_mode          读取位宽模式，参见STAT_CNT_MODE_E，0-64bit，1-128bit
* @param   start_count_id   统计起始编号，对应微码中的address
* @param   num              统计项个数
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_stat_ppu_cnt_clr(ZXIC_UINT32 dev_id,
                                     ZXIC_UINT32 queue_id, 
                                     STAT_CNT_MODE_E rd_mode,
                                     ZXIC_UINT32 start_count_id,
                                     ZXIC_UINT32 num);

/***********************************************************/
/** 清除指定vport的对应的stat统计(dtb方式)
* @param   dev_id          NP设备号
* @param   queue_id        队列号
* @param   sdt_no          流表sdt号(0~255)
* @param   vport           端口号
* @param   rd_mode         读取位宽模式，参见STAT_CNT_MODE_E，0-64bit，1-128bit
* @param   start_counter_id 统计起始编号，对应微码中的address
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_acl_stat_clr_by_vport(ZXIC_UINT32 dev_id,
                                         ZXIC_UINT32 queue_id,
                                         ZXIC_UINT32 sdt_no,
                                         ZXIC_UINT32 vport,
                                         STAT_CNT_MODE_E rd_mode,
                                         ZXIC_UINT32 start_counter_id);
#ifdef __cplusplus
}
#endif

#endif