#ifndef _DPP_DTB_TABLE_H_
#define _DPP_DTB_TABLE_H_

#ifdef __cplusplus
extern "C"{
#endif

#include "dpp_dev.h"
#include "dpp_hash.h"
#include "dpp_dtb_table_api.h"
#include "dpp_stat_api.h"

#define DPP_DISABLE (0)
#define DPP_ENABLE (1)

#define DTB_DOWN_TABLE_CMD (0)
#define DTB_DUMP_TABLE_CMD (1)

#define DTB_QUEUE_MAX (128)
#define DTB_QUEUE_ELEMENT_MAX (32)
#define DTB_DATA_SIZE_BIT (16*1024*8)
#define DPP_DTB_TABLE_DATA_BUFF_SIZE (1024*16)
#define DPP_DTB_TABLE_DUMP_INFO_BUFF_SIZE (1024*4)
#define DTB_TABLE_CMD_SIZE_BIT (128)
#define DTB_TABLE_CMD_SIZE_BYTE (16)
#define DTB_ERAM_DATA_SIZE_1b (128)
#define DTB_ERAM_DATA_SIZE_64b (128)
#define DTB_ERAM_DATA_SIZE_128b (256)
#define DTB_ERAM_ENTRY_CNT_MAX_1b (DTB_DATA_SIZE_BIT / DTB_ERAM_DATA_SIZE_1b)
#define DTB_ERAM_ENTRY_CNT_MAX_64b (DTB_DATA_SIZE_BIT / DTB_ERAM_DATA_SIZE_64b)
#define DTB_ERAM_ENTRY_CNT_MAX_128b (DTB_DATA_SIZE_BIT / DTB_ERAM_DATA_SIZE_128b)
#define DTB_ZCAM_LEN_SIZE (5) /*单位16字节*/
#define DTB_ETCAM_LEN_SIZE (6) /*单位16字节*/
#define DTB_MC_HASH_LEN_SIZE (5) /*单位16字节*/
#define DTB_ZCAM_DATA_SIZE   (ZXIC_UINT32)(64)
#define DTB_DMUP_DATA_MAX    (ZXIC_UINT32)(64*1024*1024)
// #define DTB_DUMP_DDR_ITEMS_MAX (0x10000)

#define DTB_DUMP_DDR_ITEMS_MAX (0xFFFFF)

#define DTB_TABLE_VALID  (1)
#define DTB_LEN_POS_SETP (16) /*DTB len 单位16字节 */

#define LPM_IPV4 (1)
#define LPM_IPV6 (0)
#define LPM_ENABLE (1)
#define LPM_DISABLE (0)

#define DTB_TABLE_MODE_ERAM (0)
#define DTB_TABLE_MODE_DDR (1)
#define DTB_TABLE_MODE_ZCAM (2)
#define DTB_TABLE_MODE_ETCAM (3)
#define DTB_TABLE_MODE_MC_HASH (4)

#define DTB_DUMP_MODE_ERAM (0)
#define DTB_DUMP_MODE_DDR  (1)
#define DTB_DUMP_MODE_ZCAM (2)
#define DTB_DUMP_MODE_ETCAM (3)

#define DTB_ITEM_ADD_OR_UPDATE (0)
#define DTB_ITEM_DELETE (1)

/* DTB 表信息管理 */
/*表顺序与g_dpp_dtb_table_info中顺序一致*/
typedef enum dpp_dtb_table_info_e
{
    DTB_TABLE_DDR  			= 0,
    DTB_TABLE_ERAM_1 		= 1,
	DTB_TABLE_ERAM_64 	    = 2,
	DTB_TABLE_ERAM_128      = 3,
    DTB_TABLE_ZCAM 			= 4,
    DTB_TABLE_ETCAM 		= 5,
    DTB_TABLE_MC_HASH 	    = 6,
	DTB_TABLE_ENUM_MAX
}DPP_DTB_TABLE_INFO_E;

typedef enum dpp_dtb_dump_info_e
{
    DTB_DUMP_ERAM = 0,
    DTB_DUMP_DDR  = 1,
    DTB_DUMP_ZCAM = 2,
    DTB_DUMP_ETCAM = 3,
	DTB_DUMP_ENUM_MAX
}DPP_DTB_DUMP_INFO_E;

typedef enum dpp_dtb_dump_zcam_width_e
{
    DTB_DUMP_ZCAM_128b = 0,
    DTB_DUMP_ZCAM_256b = 1,
    DTB_DUMP_ZCAM_512b = 2,
	DTB_DUMP_ZCAM_RSV  = 3,
}DPP_DTB_DUMP_ZCAM_WIDTH_E;

typedef enum dpp_dtb_dump_etcam_width_e
{
    DTB_DUMP_ETCAM_80b = 0,
    DTB_DUMP_ETCAM_160b = 1,
    DTB_DUMP_ETCAM_320b = 2,
	DTB_DUMP_ETCAM_640b  = 3,
	DTB_DUMP_ETCAM_MAX
}DPP_DTB_DUMP_ETCAM_WIDTH_E;

/* DTB下表格式字段定义 */
typedef struct dpp_dtb_ddr_table_form_t
{
	ZXIC_UINT32 valid;       /* 有效标识 1有效*/
	ZXIC_UINT32 type_mode;   /* DDR：0x1 */
    ZXIC_UINT32 rw_len;      /*数据长度 00:128  01:256  10:384  11:512*/
    ZXIC_UINT32 v46_flag;    /*1：IPV4  0:IPV6*/
    ZXIC_UINT32 lpm_wr_vld;  /*lpm表写有效标识*/
    ZXIC_UINT32 baddr;       /*表基地址*/
    ZXIC_UINT32 ecc_en;      /* ECC 使能*/
    ZXIC_UINT32 rw_addr;     /*以数据宽度为单位的index*/
}DPP_DTB_DDR_TABLE_FORM_T;

typedef struct dpp_dtb_eram_table_form_t
{
	ZXIC_UINT32 valid;        /* 有效标识 1有效*/
	ZXIC_UINT32 type_mode;    /* ERAM：0x0 */
	ZXIC_UINT32 data_mode;    /*数据长度 00:128  01:64  10:1*/
	ZXIC_UINT32 cpu_wr;       /*CPU写使能*/
	ZXIC_UINT32 cpu_rd;       /*CPU读使能*/
	ZXIC_UINT32 cpu_rd_mode;  /*CPU读模式 0：读 1：读清*/
	ZXIC_UINT32 addr;         /*访问eram 1bit为单位*/
	ZXIC_UINT32 data_h;       /*数据高32bit*/
	ZXIC_UINT32 data_l;       /*数据低32bit*/
}DPP_DTB_ERAM_TABLE_FORM_T;

typedef struct dpp_dtb_zcam_table_form_t
{
    ZXIC_UINT32 valid;         /* 有效标识 1有效*/
    ZXIC_UINT32 type_mode;     /* zcam：0x2 */
    ZXIC_UINT32 ram_reg_flag;  /* ram reg 标识 */
    ZXIC_UINT32 zgroup_id;     /* zgroup id */
    ZXIC_UINT32 zblock_id;     /* zblock id */
    ZXIC_UINT32 zcell_id;      /* zcell id */
    ZXIC_UINT32 mask;          /* 掩码 */
    ZXIC_UINT32 sram_addr;     /* ram地址 */

}DPP_DTB_ZCAM_TABLE_FORM_T;

typedef struct dpp_dtb_etcam_table_form_t
{
	ZXIC_UINT32 valid;             /* 有效标识 1有效*/
	ZXIC_UINT32 type_mode;         /* etcam：0x3 */
	ZXIC_UINT32 block_sel;         /*block索引 0 - 7*/
	ZXIC_UINT32 init_en;           /*初始化使能 高有效*/
	ZXIC_UINT32 row_or_col_msk;    /* 1 write row mask reg  0:write col mask reg*/
	ZXIC_UINT32 vben;              /* enable the valid bit addressed by addr*/
	ZXIC_UINT32 reg_tcam_flag;     /* 1:配置内部row_col_mask寄存器 0：读写tcam*/
	ZXIC_UINT32 uload;             /*使能标识删除对应addr的表项条目，（80bit为单位，含义与wr_mode一一对应）*/
	ZXIC_UINT32 rd_wr;             /*读写标志 0写 1读*/
	ZXIC_UINT32 wr_mode;           /*写入掩码，最高8bit，对应bit为1代表对应的80bit的数据*/
	ZXIC_UINT32 data_or_mask;      /*数据或掩码标志 1：写x(data),0:写y(mask)*/
	ZXIC_UINT32 addr;              /*etcam地址（0-511）*/
	ZXIC_UINT32 vbit;              /*valid bit input*/

}DPP_DTB_ETCAM_TABLE_FORM_T;

typedef struct dpp_dtb_mc_hash_table_form_t
{
	ZXIC_UINT32 valid;              /* 有效标识 1有效  */
	ZXIC_UINT32 type_mode;          /* 微码写hash 0x4 */
    ZXIC_UINT32 std_h;              /* sdt信息高32bit */
	ZXIC_UINT32 std_l;              /* sdt信息低32bit */
}DPP_DTB_MC_HASH_TABLE_FORM_T;

/* DTB DUMP表格式 */
typedef struct dpp_dtb_eram_dump_form_t
{
	ZXIC_UINT32 valid;              /* 有效标识 1有效 */
	ZXIC_UINT32 up_type;            /* 00:eram */
	ZXIC_UINT32 base_addr;          /* 128bit为单位 */
	ZXIC_UINT32 tb_depth;           /* 表项深度，每条条目位宽128bit */
	ZXIC_UINT32 tb_dst_addr_h;      /* 数据目的地址高32bit */
	ZXIC_UINT32 tb_dst_addr_l;      /* 数据目的地址低32bit */

}DPP_DTB_ERAM_DUMP_FORM_T;

typedef struct dpp_dtb_ddr_dump_form_t
{
	ZXIC_UINT32 valid;
	ZXIC_UINT32 up_type;            /* 01:ddr */
	ZXIC_UINT32 base_addr;          /* 128bit为单位 */
	ZXIC_UINT32 tb_depth;        /* 表项深度，每条条目位宽512bit */
	ZXIC_UINT32 tb_dst_addr_h;
	ZXIC_UINT32 tb_dst_addr_l;

}DPP_DTB_DDR_DUMP_FORM_T;

typedef struct dpp_dtb_zcam_dump_form_t
{
	ZXIC_UINT32 valid;
	ZXIC_UINT32 up_type;            /* 10:zcam */
	ZXIC_UINT32 zgroup_id;          /*  */
	ZXIC_UINT32 zblock_id;          /*  */
	ZXIC_UINT32 ram_reg_flag;
	ZXIC_UINT32 z_reg_cell_id;
	ZXIC_UINT32 sram_addr;
	ZXIC_UINT32 tb_depth;        /* 表项深度 */
	ZXIC_UINT32 tb_width;        /* 表项宽度 */
	ZXIC_UINT32 tb_dst_addr_h;
	ZXIC_UINT32 tb_dst_addr_l;


}DPP_DTB_ZCAM_DUMP_FORM_T;
typedef struct dpp_dtb_etcam_dump_form_t
{
	ZXIC_UINT32 valid;
	ZXIC_UINT32 up_type;         /* 11:etcam */
	ZXIC_UINT32 block_sel;       /* block num */
	ZXIC_UINT32 addr;            /* 640bit位单位 */
	ZXIC_UINT32 rd_mode;         /* 读模式，共8bit，每bit控制ram中对应位置的80bit数据是否有效*/ 
	ZXIC_UINT32 data_or_mask;    /* data：1 mask：0*/
	ZXIC_UINT32 tb_depth;        /* dump出数据深度，以640bit为单位 */
	ZXIC_UINT32 tb_width;        /* dump出数据宽度 00:80bit 01:160bit 10:320bit 11:640bit */
	ZXIC_UINT32 tb_dst_addr_h;   /* dma地址高32bit */
	ZXIC_UINT32 tb_dst_addr_l;   /* dma地址低32bit */
}DPP_DTB_ETCAM_DUMP_FORM_T;

typedef struct etcam_dump_info_t
{
	ZXIC_UINT32 block_sel;          /* block index 0-7 */
	ZXIC_UINT32 addr;               /* 单个block的RAM地址，范围0~511 640bit为单位 */
	ZXIC_UINT32 rd_mode;            /* 读模式，共8bit，每bit控制ram中对应位置的80bit数据是否有效 */
	ZXIC_UINT32 data_or_mask;       /* data：1 mask：0 参照DPP_ETCAM_DATA_TYPE_E定义*/
	ZXIC_UINT32 tb_depth;           /* dump出表深度，以640bit为单位 */
	ZXIC_UINT32 tb_width;           /* dump出数据宽度 00:80bit 01:160bit 10:320bit 11:640bit */
}DPP_ETCAM_DUMP_INFO_T;

/* 表信息结构 */
typedef struct dpp_dtb_field_t
{
    ZXIC_CHAR    *p_name;                         /* 字段名 */
    ZXIC_UINT16  lsb_pos;                         /* 最低比特位置，以寄存器列表为准*/
    ZXIC_UINT16  len;                             /* 字段长度，以比特为单位 */
}DPP_DTB_FIELD_T;

typedef struct dpp_dtb_table_t
{
    ZXIC_CHAR    *table_type;                       /* 表类型名称*/
    ZXIC_UINT32  table_no;                          /* 表编号 */
    ZXIC_UINT32  field_num;                         /* 包含的字段个数 */
    DPP_DTB_FIELD_T *p_fields;                      /* 表格式所有字段 */
}DPP_DTB_TABLE_T;

typedef struct dpp_dtb_entry_t
{
	ZXIC_UINT8 *cmd;		      /* 命令 128bit 即 16B*/
	ZXIC_UINT8 *data;	          /* 数据 */
	ZXIC_UINT32 data_in_cmd_flag; /*eram 1/64 bit模式时使用,1表示data在cmd中*/
	ZXIC_UINT32 data_size;        /* 数据长度，以字节为单位 */
}DPP_DTB_ENTRY_T;

typedef struct dpp_dtb_lpm_entry_t
{
    ZXIC_UINT32 dtb_len0;         /*存储到buff0的数据长度，当长度超过16K，则存储至buff1*/
    ZXIC_UINT8 *p_data_buff0;
    ZXIC_UINT32 dtb_len1;         /*存储到buff1的数据长度*/
    ZXIC_UINT8 *p_data_buff1;
}DPP_DTB_LPM_ENTRY_T;

typedef struct dpp_dtb_cmd_t
{
	ZXIC_UINT32 queue_id;          /*队列id*/
    ZXIC_UINT32 dtb_phy_addr_hi32; /*dtb描述符物理地址高32bit*/
    ZXIC_UINT32 dtb_phy_addr_lo32; /*dtb描述符物理地址低32bit*/
	ZXIC_UINT32 cmd_type;          /*0为流表下发命令，1为流表dump命令*/
	ZXIC_UINT32 int_enable;        /* 中断使能 */
	ZXIC_UINT32 dtb_len;           /* 指示配表内容或dump描述符长度，以16字节为单位 */
}DPP_DTB_CMD_T;


typedef struct dpp_dtb_mc_hash_key_t
{
    ZXIC_UINT32 hash_key[16];
}DPP_DTB_MC_HASH_KEY_T;

typedef struct dpp_dtb_mixed_table_t
{
    ZXIC_UINT32 down_cmd_len;        /**down描述符长度，字节为单位*/
    ZXIC_UINT32 dump_cmd_len;        /**dump描述符长度，字节为单位*/
    ZXIC_UINT32 down_buff_offset;    /**down buff 偏移，字节为单位*/
    ZXIC_UINT32 dump_buff_offset;    /**dump buff 偏移，字节为单位*/
    ZXIC_UINT8 * p_down_cmd_buff;    /**指向下表描述符空间，空间为16KB*/
    ZXIC_UINT8 * p_dump_cmd_buff;    /**指向dump描述符空间，空间为4KB*/
}DPP_DTB_MIXED_TABLE_T;

typedef struct dpp_dtb_mc_hash_entry_info_t
{
    ZXIC_UINT32 delete_en; /* delete 浣胯兘 */
    ZXIC_UINT32 dma_en;    /* dma 浣胯兘 */
    ZXIC_UINT32 *p_data;    /*hash 鏉＄洰鎸囬拡 闀垮害512bit*/    
}DPP_DTB_MC_HASH_ENTRY_INFO_T;

typedef struct dpp_dtb_stat_cfg_t
{
    ZXIC_UINT32 start_index;     /*统计起始偏移地址*/
    ZXIC_UINT32 depth;           /*统计深度 */
    STAT_CNT_MODE_E dump_mode;   /*统计模式*/    
}DPP_DTB_STAT_CFG_T;

/** dtb 中断配置
* @param   int_enable  中断使能
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_interrupt_status_set(ZXIC_UINT32 int_enable);

/** dtb 中断获取
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
ZXIC_UINT32 dpp_dtb_interrupt_status_get(ZXIC_VOID);

/** dtb cmd 大小端设置
* @param   int_enable  中断使能
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_cmd_endian_status_set(ZXIC_UINT32 endian);

/** dtb cmd 大小端获取
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_cmd_endian_status_get(ZXIC_VOID);

/***********************************************************/
/** 获取 DTB 表属性信息
* @param   table_type  DTB表类型
*
* @return  
* @remark  无
* @see
* @author  cbb      @date  2022/09/02
************************************************************/
DPP_DTB_TABLE_T *dpp_table_info_get(ZXIC_UINT32 table_type);

/***********************************************************/
/** 获取 DTB dump表属性信息
* @param   up_type  DTB表类型
*
* @return  
* @remark  无
* @see
* @author  cbb      @date  2022/09/02
************************************************************/
DPP_DTB_TABLE_T *dpp_dump_info_get(ZXIC_UINT32 up_type);

/***********************************************************/
/** dtb写下表128bit格式数据
* @param   dev_id    设备号
* @param   table_type  dtb表类型
* @param   p_cmd_data  表格式命令数据
* @param   p_cmd_buff  表格式命令数据缓存
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_write_table_cmd(ZXIC_UINT32 dev_id, 
                                   DPP_DTB_TABLE_INFO_E table_type, 
                                   ZXIC_VOID *p_cmd_data,
                                   ZXIC_VOID *p_cmd_buff);

/***********************************************************/
/** dtb写dump表128bit格式数据
* @param   dev_id    设备号
* @param   dump_type  dtb dump表类型
* @param   p_cmd_data  dump表格式命令数据
* @param   p_cmd_buff  dump表格式命令数据缓存
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_write_dump_cmd(ZXIC_UINT32 dev_id, 
                                  DPP_DTB_DUMP_INFO_E dump_type, 
                                  ZXIC_VOID *p_cmd_data,
                                  ZXIC_VOID *p_cmd_buff);
/** 写eram，输出一个entry
* @param   dev_id    芯片id
* @param   base_addr  基地址，以128bit为单位
* @param   index   条目索引，以mode为单位
* @param   wr_mode  数据位宽模式，取值参考ERAM128_OPR_MODE_E的定义
* @param   p_data   数据
* @param   p_entry   组装好的数据
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30/
************************************************************/
DPP_STATUS dpp_dtb_se_smmu0_ind_write(ZXIC_UINT32 dev_id,
                                  ZXIC_UINT32 base_addr,
                                  ZXIC_UINT32 index,
                                  ZXIC_UINT32 wrt_mode,
                                  ZXIC_UINT32 *p_data,
                                  DPP_DTB_ENTRY_T *p_entry);

/** smmu0数据组装函数,输入格式字段和数据，输出一个entry
* @param   dev_id    芯片id
* @param   mode  位宽模式
* @param   addr  1bit为单位的索引
* @param   p_data  要写入数据
* @param   p_entry  组装好的条目(已分配好空间)
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30/
************************************************************/
DPP_STATUS dpp_dtb_smmu0_write_entry_data(ZXIC_UINT32 dev_id, 
                                            ZXIC_UINT32 mode, 
                                            ZXIC_UINT32 addr, 
                                            ZXIC_UINT32 *p_data, 
                                            DPP_DTB_ENTRY_T *p_entry);

/** ddr直接表 写数据
* @param   dev_id    芯片id
* @param   base_addr  基地址，以4K*128bit为单位
* @param   rw_len  数据位宽模式，位宽 0-128bit, 1-256bit, 2-384bit, 3-512bit, 取值参考SMMU1_DDR_WRT_MODE_E的定义
* @param   index   条目索引，以mode为单位
* @param   p_data   待写入的数据指针
* @param   p_entry   组装好的条目(已分配好空间)
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30/
************************************************************/
DPP_STATUS dpp_dtb_ddr_lpm_table_data_write(ZXIC_UINT32 dev_id, 
                                        ZXIC_UINT32 base_addr,
                                        ZXIC_UINT32 rw_len,
                                        ZXIC_UINT32 ipv4_v6_flag,
                                        ZXIC_UINT32 index,
                                        ZXIC_UINT32 ecc_en,
                                        ZXIC_UINT8 *p_data, 
                                        DPP_DTB_ENTRY_T *p_entry);

/** ddr直接表 写数据
* @param   dev_id    芯片id
* @param   base_addr  基地址，以4K*128bit为单位
* @param   rw_len  数据位宽模式，位宽 0-128bit, 1-256bit, 2-384bit, 3-512bit,取值参考SMMU1_DDR_WRT_MODE_E的定义
* @param   index   条目索引，以mode为单位
* @param   ecc_en   ecc是否使能
* @param   p_data   待写入的数据指针
* @param   p_entry   组装好的条目(已分配好空间)
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30/
************************************************************/
DPP_STATUS dpp_dtb_ddr_hash_table_data_write(ZXIC_UINT32 dev_id, 
                                        ZXIC_UINT32 base_addr,
                                        ZXIC_UINT32 rw_len,
                                        ZXIC_UINT32 index,
                                        ZXIC_UINT32 ecc_en,
                                        ZXIC_UINT8 *p_data, 
                                        DPP_DTB_ENTRY_T *p_entry);

/** ddr直接表 写数据
* @param   dev_id    芯片id
* @param   base_addr  基地址，以4K*128bit为单位
* @param   rw_len  数据位宽模式，位宽 0-128bit, 1-256bit, 2-384bit, 3-512bit,取值参考SMMU1_DDR_WRT_MODE_E的定义
* @param   index   条目索引，以mode为单位
* @param   ecc_en   直接表ECC使能位
* @param   p_data   待写入的数据指针
* @param   p_entry   组装好的条目(已分配好空间)
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30/
************************************************************/
DPP_STATUS dpp_dtb_ddr_dir_table_data_write(ZXIC_UINT32 dev_id, 
                                        ZXIC_UINT32 base_addr,
                                        ZXIC_UINT32 rw_len,
                                        ZXIC_UINT32 index,
                                        ZXIC_UINT32 ecc_en,
                                        ZXIC_UINT8 *p_data, 
                                        DPP_DTB_ENTRY_T *p_entry);

/** smmu1数据组装函数,输入格式字段和数据，输出一个entry
* @param   dev_id    芯片id
* @param   rw_len  位宽
* @param   v46_flag  v4,v6flag
* @param   lpm_wr_vld  lpm表写标识
* @param   base_addr  基地址 以4K*128为单位
* @param   index  以位宽为单位的索引
* @param   ecc_en  ecc使能
* @param   p_data  要写入数据
* @param   p_entry  组装好的条目(已分配好空间)
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30/
************************************************************/
DPP_STATUS dpp_dtb_smmu1_write_entry_data(ZXIC_UINT32 dev_id, 
                                        ZXIC_UINT32 rw_len,
                                        ZXIC_UINT32 v46_flag,
                                        ZXIC_UINT32 lpm_wr_vld,
                                        ZXIC_UINT32 base_addr,
                                        ZXIC_UINT32 index,
                                        ZXIC_UINT32 ecc_en,
                                        ZXIC_UINT8 *p_data, 
                                        DPP_DTB_ENTRY_T *p_entry);

/** dtb 通道 se alg间接写zcam空间
* @param   dev_id    芯片id
* @param   addr  基地址，以4K*128bit为单位
* @param   p_data   待写入的数据指针
* @param   p_entry   组装好的条目(已分配好空间)
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30/
************************************************************/
DPP_STATUS dpp_dtb_se_alg_zcam_data_write(ZXIC_UINT32 dev_id, 
                                        ZXIC_UINT32 addr,
                                        ZXIC_UINT8 *p_data, 
                                        DPP_DTB_ENTRY_T *p_entry);

/** zcam数据组装函数,输入格式字段和数据，输出一个entry
* @param   dev_id    芯片id
* @param   reg_sram_flag  读写ZCAM寄存器/sram标志位：1'b1：读写寄存器 1'b0：读写sram
* @param   zgroup_id  
* @param   zblock_id  
* @param   zcell_id  
* @param   sram_addr  512bit为单位的地址
* @param   mask  写掩码：mask[3:0]分别对应CPU写数据的4个128-bit
*                       mask[0]对应[127:0],1'b0为不写该128-bit，1'b1为写；
*                       mask[1]对应[255:128]；
*                       mask[2]对应[383:256]；
*                       mask[3]对应[512:384]
* @param   p_data  要写入数据
* @param   p_entry  组装好的条目(已分配好空间) data空间大小为512bit
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30/
************************************************************/
DPP_STATUS dpp_dtb_zcam_write_entry_data(ZXIC_UINT32 dev_id, 
                                        ZXIC_UINT32 reg_sram_flag,
                                        ZXIC_UINT32 zgroup_id,
                                        ZXIC_UINT32 zblock_id,
                                        ZXIC_UINT32 zcell_id,
                                        ZXIC_UINT32 sram_addr,
                                        ZXIC_UINT32 mask,
                                        ZXIC_UINT8 *p_data, 
                                        DPP_DTB_ENTRY_T *p_entry);

/***********************************************************/
/** dtb 添加eTcam表条目,将etcam条目内容写入到entry中
* @param   dev_id     设备号
* @param   addr       每个block中的ram地址，位宽为8*80bit
* @param   block_idx  block编号，范围0~7
* @param   wr_mask    写表掩码，共8bit，每bit控制ram中对应位置的80bit数据是否有效
* @param   opr_type   etcam操作类型，详见 DPP_ETCAM_OPR_TYPE_E
* @param   p_entry    条目数据，data和mask
* @param   p_entry_data   组装好的数据条目(已分配好空间)
* @param   p_entry_mask   组装好的掩码条目(已分配好空间)
*
* @return
* @remark  无
* @see
* @author  王春雷      @date  2014/04/03
************************************************************/
DPP_STATUS dpp_dtb_etcam_entry_add(ZXIC_UINT32 dev_id,
                               ZXIC_UINT32 addr,
                               ZXIC_UINT32 block_idx,
                               ZXIC_UINT32 wr_mask,
                               ZXIC_UINT32 opr_type,
                               DPP_ETCAM_ENTRY_T *p_entry,
                               DPP_DTB_ENTRY_T *p_entry_data,
                               DPP_DTB_ENTRY_T *p_entry_mask
                               );

/** etcam数据组装函数,输入格式字段和数据，输出一个entry
* @param   dev_id    芯片id
* @param   block_idx         block索引 0 - 7
* @param   row_or_col_msk    1 write row mask reg  0:write col mask reg
* @param   vben              enable the valid bit addressed by addr
* @param   reg_tcam_flag     1:配置内部row_col_mask寄存器 0：读写tcam
* @param   flush             使能标识删除对应addr的表项条目，（80bit为单位，含义与wr_mode一一对应）
* @param   rd_wr             读写标志 0写 1读
* @param   wr_mode           写入掩码，最高8bit，对应bit为1代表对应的80bit的数据
* @param   data_or_mask      数据或掩码标志 1：写x(data),0:写y(mask)
* @param   ram_addr              etcam地址（0-511）
* @param   vbit              valid bit input
* @param   p_data            要写入数据
* @param   p_entry           组装好的条目(已分配好空间) data空间大小为512bit
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30/
************************************************************/
DPP_STATUS dpp_dtb_etcam_write_entry_data(ZXIC_UINT32 dev_id, 
                                        ZXIC_UINT32 block_idx,
                                        ZXIC_UINT32 row_or_col_msk,
                                        ZXIC_UINT32 vben,
                                        ZXIC_UINT32 reg_tcam_flag,
                                        ZXIC_UINT32 flush,
                                        ZXIC_UINT32 rd_wr,
                                        ZXIC_UINT32 wr_mode,
                                        ZXIC_UINT32 data_or_mask,
                                        ZXIC_UINT32 ram_addr,
                                        ZXIC_UINT32 vbit,
                                        ZXIC_UINT8 *p_data, 
                                        DPP_DTB_ENTRY_T *p_entry);

/** etcam数据组装函数,输入格式字段和数据，输出一个entry
* @param   dev_id    芯片id
* @param   sdt_high   sdt高32bit
* @param   sdt_low    sdt低32bit
* @param   delete_en  hash flag中delete使能
* @param   dma_en     hash flag中dma上送使能
* @param   p_data            key+rst
* @param   p_entry           组装好的条目(已分配好空间) data空间大小为512bit
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30/
************************************************************/
DPP_STATUS dpp_dtb_kschd_hash_write_entry_data(ZXIC_UINT32 dev_id, 
                                               ZXIC_UINT32 sdt_high, 
                                               ZXIC_UINT32 sdt_low,
                                               ZXIC_UINT32 delete_en,
                                               ZXIC_UINT32 dma_en,
                                               ZXIC_UINT32 *p_data, 
                                               DPP_DTB_ENTRY_T *p_entry);

/** 将命令+数据写入缓存中
 * 数据格式中增加，dataincmdflag，如果为1，直接拷贝cmd，不拷贝data
* @param   p_data_buff    存放数据的buff头指针
* @param   addr_offset  当前数据要写入的位置（相对于buff头的偏移）
* @param   entry  要写入缓存的数据指针
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30/
************************************************************/
DPP_STATUS dpp_dtb_data_write(ZXIC_UINT8 * p_data_buff, 
                              ZXIC_UINT32 addr_offset, 
                              DPP_DTB_ENTRY_T *entry);

/** 下表数据写入命令寄存器
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   down_table_len   数据长度,单位字节;
* @param   p_down_table_buff 下表数据缓存
* @param   p_element_id   返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_write_down_table_data(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      ZXIC_UINT32 down_table_len,
                                      ZXIC_UINT8* p_down_table_buff,
                                      ZXIC_UINT32 *p_element_id
                                      );

/** 将HASH表数据格式写入form_buff中
* @param   p_hash_cfg    hash表配置指针
* @param   p_rbkey_rtn   红黑树新节点信息指针
* @param   actu_key_size    键值长度
* @param   p_entry       dtb数据
* @param   opr_mode      配置模式 0:add/update 1:delete
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_hash_form_write(DPP_HASH_CFG *p_hash_cfg, 
                               DPP_HASH_RBKEY_INFO *p_rbkey_new,
                               ZXIC_UINT32 actu_key_size,
                               DPP_DTB_ENTRY_T *p_entry,
                               ZXIC_UINT32 opr_mode);



/***********************************************************/
/** 根据业务条目索引计算位置字段并写入缓存中
* @param   p_tbl_cfg
* @param   handle
* @param   p_block_idx
* @param   p_addr
* @param   p_wr_mask
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2014/12/23
************************************************************/
DPP_STATUS dpp_dtb_acl_etcam_form_write(ZXIC_UINT32 dev_id, 
                                DPP_ACL_TBL_CFG_T  *p_tbl_cfg, 
                                ZXIC_UINT32 handle,
                                DPP_ETCAM_DATA_TYPE_E data_or_mask,
                                DPP_ETCAM_ENTRY_T *p_entry,
                                ZXIC_UINT8 *form_buff,
                                ZXIC_UINT8 * etcam_buff
                                );															 

/** LPM表根据数据类型，提取位置信息，组装位置信息和数据
 * @param  dev_id,
* @param   dat_type         当前数据类型
* @param   p_lpm_hw_data    当前lpm位置数据指针
* @param   p_dtb_entry        需要组装的数据
* @param   entry_dtb_len      当前条目的dtb_len,16字节为单位
* @return
* @remark  
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_lpm_form_data_write(ZXIC_UINT32 dev_id,
                               ZXIC_UINT32 dat_type,
                               ZXIC_UINT32 *p_lpm_hw_data,
                               DPP_DTB_ENTRY_T * p_dtb_entry,
                               ZXIC_UINT32 *entry_dtb_len
                               );

#if ZXIC_REAL("DUMP")

/** 下发dump描述符，dump出数据 
* @param   dev_id       设备号
* @param   queue_id     队列号
* @param   element_id   元素编号
* @param   p_dump_info  dump描述符
* @param   data_len     dump出数据的长度
* @param   desc_len     dump描述符的长度
* @param   p_dump_data  dump出的数据缓存
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_write_dump_desc_info(ZXIC_UINT32 dev_id, 
                                ZXIC_UINT32 queue_id, 
                                ZXIC_UINT32 element_id,
                                ZXIC_UINT32 *p_dump_info,
                                ZXIC_UINT32 data_len,
                                ZXIC_UINT32 desc_len,
                                ZXIC_UINT32 *p_dump_data
                                );
/** smmu0 组装dump描述符函数
* @param   dev_id         设备号
* @param   base_addr      smmu0空间基地址，以128bit为单位
* @param   depth          dump的深度以128bit为单位
* @param   addr_high32    dump缓存地址高32bit
* @param   addr_low32     dump缓存地址低32bit
* @param   p_dump_info    dump描述符指针（已分配好空间128bit）
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_smmu0_dump_info_write(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 base_addr, 
                                      ZXIC_UINT32 depth,
                                      ZXIC_UINT32 addr_high32,
                                      ZXIC_UINT32 addr_low32,
                                      ZXIC_UINT32 *p_dump_info
                                      );

/** smmu1 组装dump描述符函数
* @param   dev_id         设备号
* @param   base_addr      dump基地址，以128bit为单位
* @param   depth          dump的深度以512bit为单位
* @param   addr_high32    dump缓存地址高32bit
* @param   addr_low32     dump缓存地址低32bit
* @param   p_dump_info    dump描述符指针（已分配好空间128bit）
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_smmu1_dump_info_write(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 base_addr, 
                                      ZXIC_UINT32 depth,
                                      ZXIC_UINT32 addr_high32,
                                      ZXIC_UINT32 addr_low32,
                                      ZXIC_UINT32 *p_dump_info
                                      );

/** zcam 组装dump描述符函数
* @param   dev_id         设备号
* @param   addr      zcam内部ram地址，包括掩码
* @param   tb_width  dump数据的宽度 00:128bit 01:256bit 10:512bit
* @param   depth     dump的深度以tb_width为单位
* @param   addr_high32    dump缓存地址高32bit
* @param   addr_low32     dump缓存地址低32bit
* @param   p_dump_info    dump描述符指针（已分配好空间128bit）
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_zcam_dump_info_write(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 addr, 
                                      ZXIC_UINT32 tb_width,
                                      ZXIC_UINT32 depth,
                                      ZXIC_UINT32 addr_high32,
                                      ZXIC_UINT32 addr_low32,
                                      ZXIC_UINT32 *p_dump_info
                                      );                                                                           

/** etcam 组装dump描述符函数
* @param   dev_id         设备号
* @param   p_etcam_dump_info      etcam dump内容信息
* @param   addr_high32    dump缓存地址高32bit
* @param   addr_low32     dump缓存地址低32bit
* @param   p_dump_info    dump描述符指针（已分配好空间128bit）
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_etcam_dump_info_write(ZXIC_UINT32 dev_id, 
                                      DPP_ETCAM_DUMP_INFO_T *p_etcam_dump_info,
                                      ZXIC_UINT32 addr_high32,
                                      ZXIC_UINT32 addr_low32,
                                      ZXIC_UINT32 *p_dump_info
                                      );

/** eram 生成dump entry，输入dump描述符，输出一个entry
* @param   dev_id         设备号
* @param   base_addr      smmu0空间基地址，以128bit为单位
* @param   depth          dump的深度以128bit为单位
* @param   addr_high32    dump缓存地址高32bit
* @param   addr_low32     dump缓存地址低32bit
* @param   p_entry        dump描述符指针（已分配好cmd空间128bit）
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_smmu0_dump_entry(ZXIC_UINT32 dev_id, 
                                    ZXIC_UINT32 base_addr, 
                                    ZXIC_UINT32 depth,
                                    ZXIC_UINT32 addr_high32,
                                    ZXIC_UINT32 addr_low32,
                                    DPP_DTB_ENTRY_T *p_entry);

/** etcam 生成dump entry，输入dump描述符，输出一个entry
* @param   dev_id         设备号
* @param   p_etcam_dump_info      etcam dump内容信息
* @param   addr_high32    dump缓存地址高32bit
* @param   addr_low32     dump缓存地址低32bit
* @param   p_entry        p_entry  组装dtb entry 已分配好cmd空间(128bit)
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_etcam_dump_entry(ZXIC_UINT32 dev_id, 
                                      DPP_ETCAM_DUMP_INFO_T *p_etcam_dump_info,
                                      ZXIC_UINT32 addr_high32,
                                      ZXIC_UINT32 addr_low32,
                                      DPP_DTB_ENTRY_T *p_entry);                                   

/** dtb dump DDR表项内容 返回内容以512bit为单位(最大为64MB-64B)
* @param   dev_id         设备号
* @param   queue_id       队列号
* @param   dump_phy_addr  dump内存的物理地址
* @param   dump_vir_addr  dump内存的虚拟地址
* @param   dump_mem_size  dump内存的大小,单位字节
* @param   base_addr      dump基地址，以512bit为单位
* @param   depth          dump的深度以512bit为单位，最大0xFFFFF
* @param   p_data         dump出数据缓存(512bit * depth)
* @param   element_id     返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2024/01/31
************************************************************/
DPP_STATUS dpp_dtb_se_smmu1_user_addr_dump(ZXIC_UINT32 dev_id, 
                                ZXIC_UINT32 queue_id,
                                ZXIC_UINT64 dump_phy_addr,
                                ZXIC_UINT64 dump_vir_addr,
                                ZXIC_UINT64 dump_mem_size, 
                                ZXIC_UINT32 base_addr, 
                                ZXIC_UINT32 depth,
                                ZXIC_UINT32 *p_data,
                                ZXIC_UINT32 *element_id);

/** dtb dump DDR表项内容 返回内容以512bit为单位(该函数最大dump 16KB)
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   base_addr dump基地址，以128bit为单位
* @param   depth     dump的深度以512bit为单位，最大为256
* @param   p_data    dump出数据缓存(512bit * depth)
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/11/07
************************************************************/
DPP_STATUS dpp_dtb_se_smmu1_dma_dump(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      ZXIC_UINT32 base_addr, 
                                      ZXIC_UINT32 depth,
                                      ZXIC_UINT32 *p_data,
                                      ZXIC_UINT32 *element_id
                                      );

/** dtb dump smmu0空间内容,返回内容以128bit为单位(最大为16MB-16B))
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   dump_phy_addr  dump内存的物理地址
* @param   dump_vir_addr  dump内存的虚拟地址
* @param   dump_mem_size  dump内存的大小,单位字节
* @param   base_addr 要dump的内容的基地址，以128bit为单位
* @param   depth     dump的深度以128bit为单位,最大0xFFFFF
* @param   p_data    dump出数据缓存(128bit * depth)
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_se_smmu0_user_addr_dump(ZXIC_UINT32 dev_id, 
                                           ZXIC_UINT32 queue_id, 
                                           ZXIC_UINT64 dump_phy_addr,
                                           ZXIC_UINT64 dump_vir_addr,
                                           ZXIC_UINT64 dump_mem_size,
                                           ZXIC_UINT32 base_addr, 
                                           ZXIC_UINT32 depth,
                                           ZXIC_UINT32 *p_data,
                                           ZXIC_UINT32 *element_id);

/** dtb dump smmu0空间内容(该函数最大dump16KB)
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   base_addr 要dump的内容的基地址，以128bit为单位
* @param   depth     dump的深度以128bit为单位，最大为1024
* @param   p_data    dump出数据缓存(128bit + depth)
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/11/07
************************************************************/
DPP_STATUS dpp_dtb_se_smmu0_dma_dump(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      ZXIC_UINT32 base_addr, 
                                      ZXIC_UINT32 depth,
                                      ZXIC_UINT32 *p_data,
                                      ZXIC_UINT32 *element_id
                                      );

/***********************************************************/
/** 按mode读取smmu0内容中1个条目
* @param   dev_id       设备号
* @param   queue_id     队列号
* @param   base_addr    基地址，以128bit为单位
* @param   index        条目索引,支持128、64、32和1bit的索引值
* @param   rd_mode      读eRam模式, 取值参照ERAM128_OPR_MODE_E定义，读清模式下不支持1bit模式
* @param   p_data       返回数据缓存的指针
*
* @return
* @remark  无
* @see
* @author  cbb      @date  2024/01/22
************************************************************/
DPP_STATUS dpp_dtb_se_smmu0_ind_read(ZXIC_UINT32 dev_id,
                                    ZXIC_UINT32 queue_id,
                                    ZXIC_UINT32 base_addr,
                                    ZXIC_UINT32 index,
                                    ZXIC_UINT32 rd_mode,
                                    ZXIC_UINT32 *p_data);

/** dtb通道读取stat片内内容
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   smmu0_base_addr smmu0基地址，以128bit为单位
* @param   rd_mode         读取位宽模式，参见STAT_CNT_MODE_E，0-64bit，1-128bit
* @param   index           索引，具体位宽参见rd_mode
* @param   p_data          出参，读取的数据
* @return
* @remark  无
* @see
* @author  cbb      @date  2024/01/31
************************************************************/
DPP_STATUS dpp_dtb_stat_smmu0_int_read(ZXIC_UINT32 dev_id, 
                                       ZXIC_UINT32 queue_id, 
                                       ZXIC_UINT32 smmu0_base_addr, 
                                       STAT_CNT_MODE_E rd_mode,
                                       ZXIC_UINT32 index, 
                                       ZXIC_UINT32 *p_data);

/** dtb dump eram直接表表项内容 暂时只支持128bit模式
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   sdt_no    eram表sdt表号
* @param   start_index 要dump的起始index，单位是sdt_no该表的mode
* @param   depth       dump的深度以128bit为单位
* @param   p_data      dump出数据缓存(大小128bit * depth)
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_sdt_eram_table_dump(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      ZXIC_UINT32 sdt_no, 
                                      ZXIC_UINT32 start_index, 
                                      ZXIC_UINT32 depth,
                                      ZXIC_UINT32 *p_data,
                                      ZXIC_UINT32 *element_id
                                      );

/** dtb dump eram直接表表项内容 支持64bit/128bit
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   sdt_no    eram表sdt表号
* @param   start_index 要dump的起始index，单位是sdt_no该表的mode
* @param   p_dump_data_arr 本次dump出的数据，数据格式与下表格式相同
* @param   entry_num       本次dump实际的条目数
* @param   next_start_index  下次dump是开始的index  
* @param   finish_flag       整个表dump完成标志，1表示完成，0表示未完成
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
 DPP_STATUS dpp_dtb_eram_table_dump(ZXIC_UINT32 dev_id,
                               ZXIC_UINT32 queue_id,
                               ZXIC_UINT32 sdt_no,
                               DPP_DTB_DUMP_INDEX_T start_index,
                               DPP_DTB_ERAM_ENTRY_INFO_T* p_dump_data_arr,
                               ZXIC_UINT32 *entry_num,
                               DPP_DTB_DUMP_INDEX_T *next_start_index,
                               ZXIC_UINT32 *finish_flag);

/**etcam 根据rd_mode数据提取*/
DPP_STATUS dpp_dtb_etcam_ind_data_get(ZXIC_UINT32 dev_id, ZXIC_UINT8 *p_in_data, ZXIC_UINT32 rd_mode, ZXIC_UINT8 *p_out_data);

/** dtb dump etcam直接表表项内容 支持64bit/128bit
* @param   dev_id             设备号
* @param   queue_id           队列号
* @param   sdt_no             acl表sdt表号
* @param   start_index        要dump的起始index，单位是sdt_no该表的mode
* @param   p_dump_data_arr    本次dump出的数据，数据格式与下表格式相同
* @param   entry_num          本次dump实际的条目数
* @param   next_start_index   下次dump是开始的index  
* @param   finish_flag        整个表dump完成标志，1表示完成，0表示未完成
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
 DPP_STATUS dpp_dtb_acl_table_dump(ZXIC_UINT32 dev_id,
                               ZXIC_UINT32 queue_id,
                               ZXIC_UINT32 sdt_no,
                               DPP_DTB_DUMP_INDEX_T start_index,
                               DPP_DTB_ACL_ENTRY_INFO_T* p_dump_data_arr,
                               ZXIC_UINT32 *entry_num,
                               DPP_DTB_DUMP_INDEX_T *next_start_index,
                               ZXIC_UINT32 *finish_flag);

/** DDR 直接表 dtb dump DDR直接表表项内容 暂时仅支持512bit模式
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   sdt_no    ddr直接表sdt表号
* @param   start_index 要dump的起始index，单位是sdt_no该表的mode
* @param   depth       dump的深度以128bit为单位
* @param   p_data      dump出数据缓存
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_se_ddr_dir_table_dma_dump(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      ZXIC_UINT32 sdt_no, 
                                      ZXIC_UINT32 start_index, 
                                      ZXIC_UINT32 depth,
                                      ZXIC_UINT32 *p_data,
                                      ZXIC_UINT32 *element_id
                                      );

/***********************************************************/
/** 按mode读取smmu1内容中1个条目
* @param   dev_id           设备号
* @param   queue_id         队列号
* @param   ddr_base_addr    基地址，以2K*256bit为单位
* @param   index            条目索引，以mode为单位，
* @param   ddr_mode         读ddr模式, 0-128bit,1-256bit,2-512bit
* @param   p_data           返回数据缓存的指针
*
* @return
* @remark  无
* @see
* @author  wcl      @date  2015/01/30
************************************************************/
DPP_STATUS dpp_dtb_se_smmu1_ind_read(ZXIC_UINT32 dev_id,
                                     ZXIC_UINT32 queue_id,
                                     ZXIC_UINT32 ddr_base_addr,
                                     ZXIC_UINT32 index,
                                     ZXIC_UINT32 ddr_mode,
                                     ZXIC_UINT32 *p_data);

/** dtb通道读取stat片外内容
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   smmu1_base_addr smmu1基地址，以2K*256bit为单位
* @param   rd_mode         读取位宽模式，参见STAT_CNT_MODE_E，0-64bit，1-128bit
* @param   index           索引，具体位宽参见rd_mode
* @param   p_data          出参，读取的数据
* @return
* @remark  无
* @see
* @author  cbb      @date  2024/01/31
************************************************************/
DPP_STATUS dpp_dtb_stat_smmu1_int_read(ZXIC_UINT32 dev_id, 
                                       ZXIC_UINT32 queue_id, 
                                       ZXIC_UINT32 smmu1_base_addr, 
                                       STAT_CNT_MODE_E rd_mode,
                                       ZXIC_UINT32 index, 
                                       ZXIC_UINT32 *p_data);

/** dtb dump zcam表项内容
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   addr      zcam内部ram地址，包括掩码
* @param   tb_width  dump数据的宽度
* @param   depth     dump的深度以tb_width为单位
* @param   p_data    dump出数据缓存(tb_width * depth)
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_se_zcam_dma_dump(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      ZXIC_UINT32 addr, 
                                      ZXIC_UINT32 tb_width,
                                      ZXIC_UINT32 depth,
                                      ZXIC_UINT32 *p_data,
                                      ZXIC_UINT32 *element_id);

/** dtb dump etcam表项内容
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   etcam_dump_info  etcam dump内容信息
* @param   p_data    dump出数据缓存(512bit * depth)
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_se_etcam_dma_dump(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      DPP_ETCAM_DUMP_INFO_T *etcam_dump_info,
                                      ZXIC_UINT32 *p_data,
                                      ZXIC_UINT32 *element_id);

/** dtb dump etcam 单个block内的表项内容
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   block_idx block的索引，范围0-7
* @param   index     业务条目在当前block中的相对索引，以mode为单位
* @param   mode      键值位宽模式，取值参照 DPP_ETCAM_ENTRY_MODE_E 的定义
* @param   p_data    dump出数据缓存(512bit * depth)
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_se_etcam_entry_dump(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      ZXIC_UINT32 block_idx,
                                      ZXIC_UINT32 index,
                                      ZXIC_UINT32 mode,
                                      ZXIC_UINT32 data_or_mask,
                                      ZXIC_UINT32 depth,
                                      ZXIC_UINT32 *p_data,
                                      ZXIC_UINT32 *element_id);

/** dtb根据etcam key计算dump出数据占用的数据长度
* @param   dev_id                 设备号
* @param   etcam_key_mode         etcam条目位宽，参照DPP_ETCAM_ENTRY_MODE_E定义
* @param   p_etcam_dump_len       dtb dump出etcam的数据长度(单位:字节)
* @param   p_etcam_dump_inerval   dump出etcam数据的间隔数据长度(单位：字节)
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dtb_etcam_dump_data_len(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 etcam_key_mode, 
                                   ZXIC_UINT32 *p_etcam_dump_len, 
                                   ZXIC_UINT32* p_etcam_dump_inerval);

/** dtb 从dump数据中提取到xy数据
* @param   dev_id                 设备号
* @param   p_data                 dump出的etcam data 指针
* @param   p_mask                 dump出的etcam mask 指针
* @param   p_etcam_dump_len       dtb dump出etcam的数据长度(单位:字节)
* @param   p_etcam_dump_inerval   dump出etcam数据的间隔数据长度(单位：字节)
* @param   p_entry_xy             保存etcam xy数据（已分配内存）
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_get_etcam_xy_from_dump_data(ZXIC_UINT32 dev_id, 
                                               ZXIC_UINT8 *p_data, 
                                               ZXIC_UINT8 *p_mask, 
                                               ZXIC_UINT32 etcam_dump_len,
                                               ZXIC_UINT32 etcam_dump_inerval, 
                                               DPP_ETCAM_ENTRY_T *p_entry_xy);

/** dtb dump etcam表项内容
* @param   dev_id           设备号
* @param   queue_id         队列号
* @param   block_idx        block的索引，范围0-7
* @param   addr             单个block中的RAM地址，范围0~511
* @param   rd_mode          读block RAM行的位图，共8bit，每比特表示一个RAM中的80bit
* @param   opr_type         读取的数据类型，0:data/mask格式，1：xy格式     
* @param   as_en            级联eram使能
* @param   as_eram_baddr    级联eram基地址
* @param   as_eram_index    级联eram 条目index
* @param   as_rsp_mode      级联返回位宽，参见DPP_DIAG_ERAM_MODE_E
* @param   p_entry          读取的etcam数据
* @param   p_as_rslt        读取的级联数据
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_etcam_entry_get(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 queue_id,
                                   ZXIC_UINT32 block_idx, 
                                   ZXIC_UINT32 addr,
                                   ZXIC_UINT32 rd_mode,
                                   ZXIC_UINT32 opr_type, 
                                   ZXIC_UINT32 as_en,
                                   ZXIC_UINT32 as_eram_baddr,
                                   ZXIC_UINT32 as_eram_index,
                                   ZXIC_UINT32 as_rsp_mode,//128:3   64:2
                                   DPP_ETCAM_ENTRY_T *p_entry,
                                   ZXIC_UINT8    *p_as_rslt);

/** 计算eram 128bit为单位的index
* @param   dev_id        设备号
* @param   eram_mode     eram 位宽模式
* @param   index         以eram_mode为单位的index
* @param   p_row_index   出参，行
* @param   p_col_index   出参，列
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dtb_eram_index_cal(ZXIC_UINT32 dev_id, 
                              ZXIC_UINT32 eram_mode, 
                              ZXIC_UINT32 index,
                              ZXIC_UINT32 *p_row_index, 
                              ZXIC_UINT32* p_col_index);

/** dtb dump eram直接表表项内容
* @param   dev_id       设备号
* @param   queue_id     队列号
* @param   sdt_no       eram表sdt表号
* @param   p_dump_eram_entry  eram数据结构，数据已分配相应内存
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_eram_data_get(ZXIC_UINT32 dev_id, 
                                 ZXIC_UINT32 queue_id, 
                                 ZXIC_UINT32 sdt_no, 
                                 DPP_DTB_ERAM_ENTRY_INFO_T *p_dump_eram_entry);

/** dtb dump ddr直接表表项内容
* @param   dev_id       设备号
* @param   queue_id     队列号
* @param   sdt_no       eram表sdt表号
* @param   p_dump_ddr_entry  ddr 数据结构，数据已分配相应内存
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_ddr_data_get(ZXIC_UINT32 dev_id, 
                                 ZXIC_UINT32 queue_id, 
                                 ZXIC_UINT32 sdt_no, 
                                 DPP_DTB_DDR_ENTRY_INFO_T *p_dump_ddr_entry);

/** dtb 通过key和mask获取ACL表级联结果
* @param   dev_id       设备号
* @param   queue_id     队列号
* @param   sdt_no       eram表sdt表号
* @param   p_dump_acl_entry  etcam 数据结构，数据已分配相应内存,需要输入key和mask
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_acl_data_get(ZXIC_UINT32 dev_id,
                                ZXIC_UINT32 queue_id, 
                                ZXIC_UINT32 sdt_no, 
                                DPP_DTB_ACL_ENTRY_INFO_T *p_dump_acl_entry);

/** dtb etcam 数据get接口，通过handle值获取etcam数据
* @param   dev_id       设备号
* @param   queue_id     队列号
* @param   sdt_no       eram表sdt表号
* @param   p_dump_acl_entry  etcam 数据结构，数据已分配相应内存
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_etcam_data_get(ZXIC_UINT32 dev_id,
                                  ZXIC_UINT32 queue_id, 
                                  ZXIC_UINT32 sdt_no, 
                                  DPP_DTB_ACL_ENTRY_INFO_T *p_dump_acl_entry);

ZXIC_VOID dpp_data_buff_print(ZXIC_UINT8 *buff, ZXIC_UINT32 size);                                      
ZXIC_VOID dpp_acl_data_print(ZXIC_UINT8 *p_data, ZXIC_UINT8 *p_mask, ZXIC_UINT32 etcam_mode);
ZXIC_VOID dpp_dtb_data_print(ZXIC_UINT8 *p_data, ZXIC_UINT32 len);
#endif

#if ZXIC_REAL("DTB MIXED TABLE")
/***********************************************************/
/** 获取dtb 混合下表管理结构
* @param   dev_id               设备号，支持多芯片
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
DPP_DTB_MIXED_TABLE_T *dpp_dtb_mixed_table_mgr_get(ZXIC_UINT32 dev_id);

/***********************************************************/
/** dtb 混合下表初使能
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_mixed_table_enable(void);

/***********************************************************/
/** dtb 混合下表关闭
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_mixed_table_disable(void);

/***********************************************************/
/** dtb 混合下表功能状态获取
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_mixed_table_state_get(void);

/***********************************************************/
/** dtb 下表结果不检查
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_down_check_disable(void);

/***********************************************************/
/** dtb 下表结果检查
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_down_check_enable(void);

/***********************************************************/
/** dtb 下表结果检查设置获取
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_down_check_state_get(void);

/***********************************************************/
/** dtb 混合下表初始化
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_mixed_table_init(ZXIC_UINT32 dev_id);

/***********************************************************/
/** dtb 混合下表结构体释放
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_mixed_table_release(ZXIC_UINT32 dev_id);

/***********************************************************/
/** dtb 混合下表结构体内容清空
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_mixed_table_info_clr(ZXIC_UINT32 dev_id);

/***********************************************************/
/** dtb 混合下表结构体内容打印
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_mixed_table_info_prt(ZXIC_UINT32 dev_id);

/***********************************************************/
/** dtb 混合下表数据保存
* @param   p_down_data     下表数据指针         
* @param   down_data_len   下表数据长度，单位：字节    
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_mixed_down_data_saved(ZXIC_UINT32 dev_id, 
                                          ZXIC_UINT8 * p_down_data,
                                          ZXIC_UINT32 down_data_len);

/***********************************************************/
/** dtb 混合下表处理函数
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_mixed_down_table_process(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id, ZXIC_UINT32 *element_id);


/***********************************************************/
/** dtb 混合dump数据保存
* @param   p_dump_cmd_data   dump描述符指针         
* @param   dump_cmd_len      dump描述符长度，单位：字节    
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_mixed_dump_cmd_info_saved(ZXIC_UINT32 dev_id, 
                                          ZXIC_UINT8 *p_dump_cmd_data,
                                          ZXIC_UINT32 dump_cmd_len);

/***********************************************************/
/** dtb 混合dump处理函数
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  cbb      @date  2023/05/24
************************************************************/
ZXIC_UINT32 dpp_dtb_mixed_dump_table_process(ZXIC_UINT32 dev_id, 
                                             ZXIC_UINT32 queue_id, 
                                             ZXIC_UINT32 *p_element_id);

/** dtb写smmu0中的数据，数据长度不限
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   smmu0_base_addr smmu0基地址，以128bit为单位
* @param   smmu0_wr_mode   smmu0写模式，参考DPP_ERAM128_OPR_MODE_E，仅支持128bit、64bit、1bit模式
* @param   entry_num       下发的条目数
* @param   p_entry_arr     待下发表项内容结构体数组指针
* @param   element_id      返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2024/01/04
************************************************************/
DPP_STATUS dpp_dtb_smmu0_data_write(ZXIC_UINT32 dev_id,
                                    ZXIC_UINT32 queue_id,
                                    ZXIC_UINT32 smmu0_base_addr,
                                    ZXIC_UINT32 smmu0_wr_mode,
                                    ZXIC_UINT32 entry_num,
                                    DPP_DTB_ERAM_ENTRY_INFO_T *p_entry_arr,
                                    ZXIC_UINT32 *element_id);

/** dtb写eRam表
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   sdt_no    eram表sdt表号
* @param   entry_cnt 下发的条目数
* @param   p_entry_arr   待下发表项内容结构体数组指针
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_eram_dma_write(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      ZXIC_UINT32 sdt_no, 
                                      ZXIC_UINT32 entry_num, 
                                      DPP_DTB_ERAM_ENTRY_INFO_T *p_entry_arr,
                                      ZXIC_UINT32 *element_id);

/** dtb flush smmu0中的数据,大数据量
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   smmu0_base_addr smmu0基地址，以128bit为单位
* @param   smmu0_wr_mode   smmu0写模式，参考DPP_ERAM128_OPR_MODE_E，仅支持128bit、64bit、1bit模式
* @param   start_index     flush开始的条目
* @param   entry_num       下发的条目数
* @param   p_entry_arr     待下发表项内容结构体数组指针
* @param   element_id      返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2024/01/04
************************************************************/
DPP_STATUS dpp_dtb_smmu0_flush(ZXIC_UINT32 dev_id,
                                     ZXIC_UINT32 queue_id,
                                     ZXIC_UINT32 smmu0_base_addr,
                                     ZXIC_UINT32 smmu0_wr_mode,
                                     ZXIC_UINT32 start_index,
                                     ZXIC_UINT32 entry_num,
                                     ZXIC_UINT32 *element_id);

/** dtb写ddr中的数据，数据长度在16K范围内
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   ddr_base_addr   ddr基地址，以4K*128bit为单位
* @param   ddr_wr_mode     ddr写模式 0-128bit, 1-256bit, 2-384bit, 3-512bit,取值参考SMMU1_DDR_WRT_MODE_E的定义
* @param   ddr_ecc_en      ddr ECC使能
* @param   entry_num       下发的条目数
* @param   p_entry_arr     待下发表项内容结构体数组指针
* @param   element_id      返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_smmu1_data_write_cycle(ZXIC_UINT32 dev_id, 
                                          ZXIC_UINT32 queue_id, 
                                          ZXIC_UINT32 ddr_base_addr, 
                                          ZXIC_UINT32 ddr_wr_mode, 
                                          ZXIC_UINT32 ddr_ecc_en,
                                          ZXIC_UINT32 entry_num, 
                                          DPP_DTB_DDR_ENTRY_INFO_T *p_entry_arr,
                                          ZXIC_UINT32 *element_id);

/** dtb写ddr中的数据，数据长度无限制
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   ddr_base_addr   ddr基地址，以4K*128bit为单位
* @param   ddr_wr_mode     ddr写模式 0-128bit, 1-256bit, 2-384bit, 3-512bit,取值参考SMMU1_DDR_WRT_MODE_E的定义
* @param   ddr_ecc_en      ddr ECC使能
* @param   entry_num       下发的条目数
* @param   p_entry_arr     待下发表项内容结构体数组指针
* @param   element_id      返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_smmu1_data_write(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 queue_id, 
                                   ZXIC_UINT32 ddr_base_addr, 
                                   ZXIC_UINT32 ddr_wr_mode, 
                                   ZXIC_UINT32 ddr_ecc_en,
                                   ZXIC_UINT32 entry_num, 
                                   DPP_DTB_DDR_ENTRY_INFO_T *p_entry_arr,
                                   ZXIC_UINT32 *element_id);

/** dtb写DDT直接表
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   sdt_no    eram表sdt表号
* @param   entry_cnt 下发的条目数
* @param   p_entry_arr   待下发表项内容结构体数组指针
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_ddr_dma_write(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      ZXIC_UINT32 sdt_no, 
                                      ZXIC_UINT32 entry_num, 
                                      DPP_DTB_DDR_ENTRY_INFO_T *p_entry_arr,
                                      ZXIC_UINT32 *element_id
                                      );

/** dtb写HASH表,在插入条目时如果冲突，则对冲突条目进行记录
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   sdt_no    hash表sdt表号
* @param   entry_cnt 下发的条目数
* @param   p_entry_arr   待下发表项内容结构体数组指针
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无 是否是有一个写不成功就返回，还是继续进行下一个条目并记录错误的条目
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_hash_dma_insert(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 queue_id, 
                                   ZXIC_UINT32 sdt_no, 
                                   ZXIC_UINT32 entry_num, 
                                   DPP_DTB_HASH_ENTRY_INFO_T *p_arr_hash_entry,
                                   ZXIC_UINT32 *element_id);

/** dtb删除HASH表
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   sdt_no    hash表sdt表号
* @param   entry_cnt 删除的条目数
* @param   p_entry_arr   待下发表项内容结构体数组指针
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无 
* @see
* @author        @date  2023/03/14
************************************************************/
DPP_STATUS dpp_dtb_hash_dma_delete_cycle(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 queue_id, 
                                   ZXIC_UINT32 sdt_no, 
                                   ZXIC_UINT32 entry_num, 
                                   DPP_DTB_HASH_ENTRY_INFO_T *p_arr_hash_entry,
                                   ZXIC_UINT32 *element_id);

/** dtb写ACL表 （SPECIFY模式，条目中指定handle，不支持级联）
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   sdt_no    ACL表sdt表号
* @param   entry_cnt 下发的条目数
* @param   p_entry_arr   待下发表项内容结构体数组指针
* @param   element_id   返回下表使用的元素id
* @param   as_element_id   返回级联结果使用的元素id
* @return
* @remark  无 是否是有一个写不成功就返回，还是继续进行下一个条目并记录错误的条目
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_acl_dma_insert(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      ZXIC_UINT32 sdt_no, 
                                      ZXIC_UINT32 entry_num, 
                                      DPP_DTB_ACL_ENTRY_INFO_T *p_acl_entry_arr,
                                      ZXIC_UINT32 *element_id
                                      );

/** dtb模拟微码写hash表 
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   sdt_no    hash表sdt表号
* @param   entry_cnt 下发的条目数
* @param   p_entry_arr   待下发表项内容结构体数组指针
* @param   element_id   返回下表使用的元素id
* @return
* @remark  无 是否是有一个写不成功就返回，还是继续进行下一个条目并记录错误的条目
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_dma_kschd_hash_insert(ZXIC_UINT32 dev_id, 
                                      ZXIC_UINT32 queue_id, 
                                      ZXIC_UINT32 sdt_no, 
                                      ZXIC_UINT32 entry_num, 
                                      DPP_DTB_MC_HASH_ENTRY_INFO_T *p_arr_mc_hash_entry,
                                      ZXIC_UINT32 *element_id
                                      );

/** dtb写LPM路由表接口,因为LPM下表是需要搬移的数据较多，只支持1个表项逐条下发
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   sdt_no    hash表sdt表号
* @param   p_entry   待下发表项内容结构体数组指针 v4/6结构体 DPP_ROUTE_ENTRY_IPV4_T / DPP_ROUTE_IPV6_KEY_T
* @return
* @remark  无 是否是有一个写不成功就返回，还是继续进行下一个条目并记录错误的条目
* @see
* @author  cbb      @date  2022/08/30
************************************************************/
DPP_STATUS dpp_dtb_route_dma_insert(ZXIC_UINT32 dev_id, 
                                    ZXIC_UINT32 queue_id, 
                                    ZXIC_UINT32 sdt_no, 
                                    ZXIC_VOID *p_entry,
                                    ZXIC_UINT32 *element_id
                                    );

/** dtb删除LPM路由表接口,因为LPM下表是需要搬移的数据较多，只支持1个表项逐条下发
* @param   dev_id    设备号
* @param   queue_id  队列号
* @param   sdt_no    hash表sdt表号
* @param   p_entry   待下发表项内容结构体数组指针 v4/6结构体 DPP_ROUTE_ENTRY_IPV4_T / DPP_ROUTE_IPV6_KEY_T
* @return
* @remark  无 
* @see
* @author        @date  2023/03/14
************************************************************/
DPP_STATUS dpp_dtb_route_dma_delete(ZXIC_UINT32 dev_id, 
                                    ZXIC_UINT32 queue_id, 
                                    ZXIC_UINT32 sdt_no, 
                                    ZXIC_VOID *p_entry,
                                    ZXIC_UINT32 *element_id
                                    );     

/***********************************************************/
/** 根据eram mode计算dtb长度
* @param   mode  eram模式   
* @return  dtb长度(单位16B)
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_eram_dtb_len_get(ZXIC_UINT32 mode);

/***********************************************************/
/** dtb eram一条表项描述符生成
* @param   dev_id       设备号，支持多芯片   
* @param   sdt_no       0~255 
* @param   del_en       1：表项删除 0：添加或者更新
* @param   pData        eram表项结构体信息
* @param   p_dtb_len    出参，返回描述符长度(单位16B)
* @param   p_dtb_one_entry    出参，返回描述符信息
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_eram_one_entry(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 sdt_no,
                                   ZXIC_UINT32 del_en,
                                   ZXIC_VOID *pData,
                                   ZXIC_UINT32 *p_dtb_len, 
                                   DPP_DTB_ENTRY_T *p_dtb_one_entry);  

/***********************************************************/
/** dtb ddr一条表项描述符生成
* @param   dev_id       设备号，支持多芯片   
* @param   sdt_no       0~255 
* @param   del_en       1：表项删除 0：添加或者更新 
* @param   pData        ddr表项结构体信息
* @param   p_dtb_len    出参，返回描述符长度(单位16B)
* @param   p_dtb_one_entry    出参，返回描述符信息
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_ddr_one_entry(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 sdt_no,
                                   ZXIC_UINT32 del_en,
                                   ZXIC_VOID *pData,
                                   ZXIC_UINT32 *p_dtb_len, 
                                   DPP_DTB_ENTRY_T *p_dtb_one_entry);

/***********************************************************/
/** 根据输入的码流和sdt号生成一条dtb hash描述符
* @param   dev_id       设备号，支持多芯片   
* @param   sdt_no       0~255 
* @param   p_hash_entry  hash表项码流，包括键值和rst
* @param   p_dtb_one_entry    出参，返回描述符信息
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_hash_insert(ZXIC_UINT32 dev_id, 
                                ZXIC_UINT32 sdt_no,
                                DPP_HASH_ENTRY *p_hash_entry,
                                DPP_DTB_ENTRY_T *p_dtb_one_entry);

/***********************************************************/
/** 根据输入的码流和sdt号生成一条dtb hash描述符(删除hash表项)
* @param   dev_id       设备号，支持多芯片   
* @param   sdt_no       0~255 
* @param   p_hash_entry  hash表项码流，包括键值和rst
* @param   p_dtb_one_entry    出参，返回描述符信息
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_hash_delete(ZXIC_UINT32 dev_id, 
                                ZXIC_UINT32 sdt_no,
                                DPP_HASH_ENTRY *p_hash_entry,
                                DPP_DTB_ENTRY_T *p_dtb_one_entry);

/***********************************************************/
/** 生成一条dtb hash描述符
* @param   dev_id       设备号，支持多芯片   
* @param   sdt_no       0~255 
* @param   del_en       1：表项删除 0：添加或者更新 
* @param   pData        eram表项结构体信息
* @param   p_dtb_len    出参，返回描述符长度(单位16B)
* @param   p_dtb_one_entry    出参，返回描述符信息
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_hash_one_entry(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 sdt_no,
                                   ZXIC_UINT32 del_en,
                                   ZXIC_VOID *pData,
                                   ZXIC_UINT32 *p_dtb_len, 
                                   DPP_DTB_ENTRY_T *p_dtb_one_entry);

/***********************************************************/
/** 根据eram级联模式获取eram读写模式
* @param   as_mode  eram模式   
* @return  dtb长度(单位16B)
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_eram_opr_mode_get(ZXIC_UINT32 as_mode);

/***********************************************************/
/** 根据输入的码流和sdt号生成一条dtb acl描述符
* @param   dev_id       设备号，支持多芯片   
* @param   sdt_no       0~255 
* @param   p_hash_entry  hash表项码流，包括键值和rst
* @param   p_dtb_one_entry    出参，返回描述符信息
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_acl_insert(ZXIC_UINT32 dev_id, 
                                ZXIC_UINT32 sdt_no,
                                DPP_ACL_ENTRY_EX_T *p_acl_entry,
                                DPP_DTB_ENTRY_T *p_dtb_data_entry,
                                DPP_DTB_ENTRY_T *p_dtb_mask_entry,
                                DPP_DTB_ENTRY_T *p_dtb_as_entry);

/***********************************************************/
/** 根据输入的码流和sdt号生成一条dtb acl描述符(删除表项)
* @param   dev_id       设备号，支持多芯片   
* @param   sdt_no       0~255 
* @param   p_hash_entry  hash表项码流，包括键值和rst
* @param   p_dtb_one_entry    出参，返回描述符信息
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_acl_delete(ZXIC_UINT32 dev_id, 
                                ZXIC_UINT32 sdt_no,
                                DPP_ACL_ENTRY_EX_T *p_acl_entry,
                                DPP_DTB_ENTRY_T *p_dtb_data_entry,
                                DPP_DTB_ENTRY_T *p_dtb_mask_entry,
                                DPP_DTB_ENTRY_T *p_dtb_as_entry);

/***********************************************************/
/** 生成dtb acl描述符，并写入缓存
* @param   dev_id       设备号，支持多芯片   
* @param   sdt_no       0~255 
* @param   del_en       1：表项删除 0：添加或者更新 
* @param   pData        acl表项结构体信息
* @param   p_dtb_len    出参，返回描述符长度(单位16B)
* @param   p_dtb_one_entry    出参，返回描述符信息
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_acl_one_entry(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 sdt_no,
                                   ZXIC_UINT32 del_en,
                                   ZXIC_VOID *pData,
                                   ZXIC_UINT32 *p_dtb_len, 
                                   ZXIC_UINT8 *p_data_buff);

/***********************************************************/
/** 根据输入的码流和sdt号生成一条dtb lpm描述符
* @param   dev_id       设备号，支持多芯片   
* @param   sdt_no       0~255 
* @param   pData        lpm表项信息
* @param   p_lpm_entry  出参，返回描述符信息
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_lpm_insert(ZXIC_UINT32 dev_id, 
                                ZXIC_UINT32 sdt_no,
                                ZXIC_VOID *pData,
                                DPP_DTB_LPM_ENTRY_T *p_lpm_entry);

/***********************************************************/
/** 根据输入的码流和sdt号生成一条dtb lpm描述符(删除表项)
* @param   dev_id       设备号，支持多芯片   
* @param   sdt_no       0~255 
* @param   pData        lpm表项信息
* @param   p_lpm_entry  出参，返回描述符长度(单位16B)和描述符信息
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_lpm_delete(ZXIC_UINT32 dev_id, 
                                ZXIC_UINT32 sdt_no,
                                ZXIC_VOID *pData,
                                DPP_DTB_LPM_ENTRY_T *p_lpm_entry);
                               
/***********************************************************/
/** 生成dtb lpm描述符,并写入缓存
* @param   dev_id       设备号，支持多芯片   
* @param   sdt_no       0~255 
* @param   pData        acl表项结构体信息
* @param   p_lpm_entry  出参，返回描述符长度(单位16B)和描述符信息
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_lpm_one_entry(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 sdt_no,
                                   ZXIC_UINT32 del_en,
                                   ZXIC_VOID *pData,
                                   DPP_DTB_LPM_ENTRY_T *p_lpm_entry);


/***********************************************************/
/** 根据键值查找hash表
* @param   dev_id       设备号，支持多芯片 
* @param   queue_id     队列id 
* @param   sdt_no       0~255 
* @param   p_dtb_hash_entry    出参，返回描述符信息
* @param   srh_mode   查找模式软件or硬件 DPP_HASH_SRH_MODE
* @return  
* @remark  无
* @see     
* @author  cq      @date  2023/08/02
************************************************************/
ZXIC_UINT32 dpp_dtb_hash_data_get(ZXIC_UINT32 dev_id, 
                                ZXIC_UINT32 queue_id,
                                ZXIC_UINT32 sdt_no,
                                DPP_DTB_HASH_ENTRY_INFO_T *p_dtb_hash_entry,
                                ZXIC_UINT32 srh_mode);

ZXIC_UINT32 dpp_dtb_hash_ddr_get(ZXIC_UINT32 dev_id, 
                                 ZXIC_UINT32 queue_id,
                                 HASH_ENTRY_CFG  *p_hash_entry_cfg,
                                 DPP_HASH_ENTRY  *p_hash_entry,
                                 ZXIC_UINT32 srh_mode,
                                 ZXIC_UINT8 *p_srh_succ);

ZXIC_UINT32 dpp_dtb_hash_zcam_get(ZXIC_UINT32 dev_id, 
                                 ZXIC_UINT32 queue_id,
                                 HASH_ENTRY_CFG  *p_hash_entry_cfg,
                                 DPP_HASH_ENTRY  *p_hash_entry,
                                 ZXIC_UINT32 srh_mode,
                                 ZXIC_UINT8 *p_srh_succ);

DPP_STATUS dpp_dtb_hash_zcam_get_hardware(ZXIC_UINT32 dev_id, 
                                 ZXIC_UINT32 queue_id,
                                 HASH_ENTRY_CFG  *p_hash_entry_cfg,
                                 DPP_HASH_ENTRY  *p_hash_entry,
                                 ZXIC_UINT8 *p_srh_succ);

ZXIC_UINT32 dpp_dtb_hash_ddr_get_hardware(ZXIC_UINT32 dev_id, 
                                 ZXIC_UINT32 queue_id,
                                 HASH_ENTRY_CFG  *p_hash_entry_cfg,
                                 DPP_HASH_ENTRY  *p_hash_entry,
                                 ZXIC_UINT8 *p_srh_succ);

DPP_STATUS dpp_dtb_hash_get_software(ZXIC_UINT32 dev_id, 
                                 HASH_ENTRY_CFG  *p_hash_entry_cfg,
                                 DPP_HASH_ENTRY  *p_hash_entry,
                                 ZXIC_UINT8 *p_srh_succ);
#if 0
ZXIC_UINT32 dpp_dtb_hash_zcell_get(ZXIC_UINT32 dev_id, 
                                 ZXIC_UINT32 queue_id,
                                 HASH_ENTRY_CFG  *p_hash_entry_cfg,
                                 DPP_HASH_ENTRY  *p_hash_entry,
                                 ZXIC_UINT8 *p_srh_succ);

ZXIC_UINT32 dpp_dtb_hash_zreg_get(ZXIC_UINT32 dev_id, 
                                 ZXIC_UINT32 queue_id,
                                 HASH_ENTRY_CFG  *p_hash_entry_cfg,
                                 DPP_HASH_ENTRY  *p_hash_entry,
                                 ZXIC_UINT8 *p_srh_succ);
#endif
ZXIC_UINT32 dpp_dtb_hash_data_parse(ZXIC_UINT32 item_type, 
                                   ZXIC_UINT32 key_by_size,
                                   DPP_HASH_ENTRY *p_entry,
                                   ZXIC_UINT8  *p_item_data,
                                   ZXIC_UINT32  *p_data_offset);

DPP_STATUS dpp_dtb_hash_software_item_check(DPP_HASH_ENTRY *p_entry,
                                        ZXIC_UINT32 key_by_size,
                                        ZXIC_UINT32 rst_by_size,
                                        SE_ITEM_CFG *p_item_info);

DPP_STATUS dpp_dtb_smmu1_flush(ZXIC_UINT32 dev_id, 
                               ZXIC_UINT32 queue_id, 
                               ZXIC_UINT32 ddr_base_addr, 
                               ZXIC_UINT32 ddr_wr_mode, 
                               ZXIC_UINT32 ddr_ecc_en,
                               ZXIC_UINT32 entry_num, 
                               ZXIC_UINT32 *element_id);

DPP_STATUS dpp_dtb_specify_zcam_space_clr(DPP_SE_CFG *p_se_cfg,
                                           ZXIC_UINT32 queue_id,
                                           ZXIC_UINT32 fun_id,
                                           ZXIC_UINT32 bulk_id);

DPP_STATUS dpp_dtb_zcam_space_clr(DPP_SE_CFG *p_se_cfg,
                                           ZXIC_UINT32 queue_id,
                                           ZXIC_UINT32 fun_id);

DPP_STATUS dpp_dtb_specify_zblock_clr(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id, ZXIC_UINT32 zblock_id);

DPP_STATUS dpp_dtb_hash_zreg_clr(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id, ZXIC_UINT32 zblk_id, ZXIC_UINT32 zreg_id);

DPP_STATUS dpp_dtb_hash_zcell_clr(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id, ZXIC_UINT32 zcell_id);

DPP_STATUS dpp_dtb_hash_zcell_range_clr(ZXIC_UINT32 dev_id, 
                                        ZXIC_UINT32 queue_id, 
                                        ZXIC_UINT32 zcell_id, 
                                        ZXIC_UINT32 start_index, 
                                        ZXIC_UINT32 num);
                                        
DPP_STATUS dpp_dtb_hash_lpm_destroy(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id, ZXIC_UINT32 flush_mode);

DPP_STATUS dpp_dtb_hash_destroy(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id, ZXIC_UINT32 flush_mode);

DPP_STATUS dpp_dtb_one_hash_destroy(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id, ZXIC_UINT32 hash_id, ZXIC_UINT32 flush_mode);

DPP_STATUS dpp_dtb_hash_all_entry_delete(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id, ZXIC_UINT32 hash_id, ZXIC_UINT32 flush_mode);

DPP_STATUS dpp_dtb_dump_hash_parse(HASH_ENTRY_CFG *p_hash_entry_cfg,
                                    ZXIC_UINT32 item_type,
                                    ZXIC_UINT8 *pdata,
                                    ZXIC_UINT32 dump_len,
                                    ZXIC_UINT8 *pOutData,
                                    ZXIC_UINT32 *p_item_num);

DPP_STATUS dpp_dtb_dump_addr_set(ZXIC_UINT32 dev_id,
                                 ZXIC_UINT32 queue_id,
                                 ZXIC_UINT32 sdt_no, 
                                 ZXIC_UINT32 *element_id);

DPP_STATUS dpp_dtb_hash_table_zcam_dump(ZXIC_UINT32 dev_id,
                                        ZXIC_UINT32 queue_id,
                                        ZXIC_UINT32 sdt_no, 
                                        ZXIC_UINT32 zblock_num,
                                        ZXIC_UINT32 zblock_array[SE_ZBLK_NUM],
                                        ZXIC_UINT8  *p_data,
                                        ZXIC_UINT32 *p_dump_len);

DPP_STATUS dpp_dtb_ddr_data_dump(ZXIC_UINT32 dev_id,
                                    ZXIC_UINT32 queue_id,
                                    ZXIC_UINT32 sdt_no,
                                    ZXIC_UINT32 base_addr,
                                    ZXIC_UINT32 depth,
                                    ZXIC_UINT8  *p_data);

DPP_STATUS dpp_dtb_hash_table_dump(ZXIC_UINT32 dev_id,
                               ZXIC_UINT32 queue_id,
                               ZXIC_UINT32 sdt_no,
                               DPP_DTB_DUMP_INDEX_T startIndex,
                               ZXIC_UINT8* pDumpData,
                               ZXIC_UINT32 *entryNum,
                               DPP_DTB_DUMP_INDEX_T *nextStartIndex,
                               ZXIC_UINT32 *finishFlag);

DPP_STATUS dpp_dtb_hash_table_only_zcam_dump(ZXIC_UINT32 dev_id,
                                            ZXIC_UINT32 queue_id,
                                            ZXIC_UINT32 sdt_no,
                                            ZXIC_UINT8* pDumpData,
                                            ZXIC_UINT32 *entryNum);

DPP_STATUS dpp_dtb_dump_ddr_parse(ZXIC_UINT32 item_width,
                                  ZXIC_UINT32 start_index,
                                  ZXIC_UINT8 *pdata,
                                  ZXIC_UINT32 dump_len,
                                  ZXIC_UINT8 *pOutData,
                                  ZXIC_UINT32 *p_item_num);

DPP_STATUS dpp_dtb_ddr_table_dump(ZXIC_UINT32 dev_id,
                               ZXIC_UINT32 queue_id,
                               ZXIC_UINT32 sdt_no,
                               DPP_DTB_DUMP_INDEX_T startIndex,
                               ZXIC_UINT8* pDumpData,
                               ZXIC_UINT32 *entryNum,
                               DPP_DTB_DUMP_INDEX_T *nextStartIndex,
                               ZXIC_UINT32 *finishFlag);

DPP_STATUS dpp_dtb_eram_table_flush(ZXIC_UINT32 dev_id,
                               ZXIC_UINT32 queue_id,
                               ZXIC_UINT32 sdt_no);

DPP_STATUS dpp_dtb_hash_table_flush(ZXIC_UINT32 dev_id,
                               ZXIC_UINT32 queue_id,
                               ZXIC_UINT32 sdt_no,
                               ZXIC_UINT32 flush_mode);

DPP_STATUS dpp_dtb_ddr_table_flush(ZXIC_UINT32 dev_id,
                                   ZXIC_UINT32 queue_id,
                                   ZXIC_UINT32 sdt_no,
                                   ZXIC_UINT32 flush_mode);

DPP_STATUS dpp_dtb_etcam_table_flush(ZXIC_UINT32 dev_id,
                                     ZXIC_UINT32 queue_id,
                                     ZXIC_UINT32 sdt_no);

DPP_STATUS dpp_dtb_lpm_offline_delete(ZXIC_UINT32 dev_id,
                                      ZXIC_UINT32 queue_id,
                                      ZXIC_UINT32 sdt_no,
                                      ZXIC_UINT32 flush_mode);

DPP_STATUS dpp_dtb_lpm_offline_delete_as(ZXIC_UINT32 dev_id,
                                         ZXIC_UINT32 queue_id,
                                         ZXIC_UINT32 as_type,
                                         ZXIC_UINT32 alg_lpm_type,
                                         ZXIC_UINT32 table_depth,
                                         ZXIC_UINT32 flush_mode);
BOOLEAN dpp_hash_sdt_partner_valid( ZXIC_UINT32 dev_id, ZXIC_UINT32 sdt_no,ZXIC_UINT32 sdt_partner,ZXIC_UINT8 *p_key);

DPP_STATUS dpp_dtb_hash_del_software(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 queue_id, 
                                   ZXIC_UINT32 sdt_no, 
                                   ZXIC_UINT32 entry_num, 
                                   DPP_DTB_HASH_ENTRY_INFO_T *p_arr_hash_entry,
                                   ZXIC_UINT32 *element_id);

ZXIC_UINT32 dpp_ddr_index_calc(ZXIC_UINT32 index,
                               ZXIC_UINT32 width_mode,
                               ZXIC_UINT32 key_type,
                               ZXIC_UINT32 byte_offset);

DPP_STATUS dpp_dtb_hash_zcam_del_hw(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 queue_id, 
                                   ZXIC_UINT32 sdt_no, 
                                   ZXIC_UINT32 entry_num, 
                                   DPP_DTB_HASH_ENTRY_INFO_T *p_arr_hash_entry,
                                   ZXIC_UINT32 *element_id);

DPP_STATUS dpp_dtb_hash_zcam_del_one_hw(ZXIC_UINT32 dev_id, 
                                 ZXIC_UINT32 queue_id,
                                 HASH_ENTRY_CFG  *p_hash_entry_cfg,
                                 DPP_HASH_ENTRY  *p_hash_entry,
                                 DPP_DTB_ENTRY_T *p_entry,
                                 ZXIC_UINT8 *p_srh_succ);

DPP_STATUS dpp_dtb_hash_ddr_del_hw(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 queue_id, 
                                   ZXIC_UINT32 sdt_no, 
                                   ZXIC_UINT32 entry_num, 
                                   DPP_DTB_HASH_ENTRY_INFO_T *p_arr_hash_entry,
                                   ZXIC_UINT32 *element_id);

DPP_STATUS dpp_dtb_hash_ddr_del_hw_cycle(ZXIC_UINT32 dev_id, 
                                   ZXIC_UINT32 queue_id, 
                                   HASH_ENTRY_CFG *p_hash_entry_cfg, 
                                   ZXIC_UINT32 entry_num, 
                                   DPP_DTB_HASH_ENTRY_INFO_T *p_arr_hash_entry,
                                   ZXIC_UINT32 *element_id);

ZXIC_UINT32 dpp_dtb_hash_ddr_del_one_hw(ZXIC_UINT32 dev_id, 
                                 ZXIC_UINT32 queue_id,
                                 HASH_ENTRY_CFG  *p_hash_entry_cfg,
                                 DPP_HASH_ENTRY  *p_hash_entry,
                                 DPP_DTB_ENTRY_T *p_entry,
                                 ZXIC_UINT8 *p_srh_succ);

DPP_STATUS dpp_dtb_hash_ddr_spec_clear(ZXIC_UINT32 dev_id, ZXIC_UINT32 queue_id, HASH_ENTRY_CFG *p_hash_entry_cfg, ZXIC_UINT32 flush_mode);

DPP_STATUS dpp_dtb_hash_offline_zcam_delete(ZXIC_UINT32 dev_id,
                                        ZXIC_UINT32 queue_id,
                                        ZXIC_UINT32 sdt_no);

DPP_STATUS dpp_dtb_hash_offline_ddr_delete(ZXIC_UINT32 dev_id,
                                        ZXIC_UINT32 queue_id,
                                        ZXIC_UINT32 sdt_no,
                                        ZXIC_UINT32 flush_mode);

DPP_STATUS dpp_dtb_hash_online_zcam_delete(ZXIC_UINT32 dev_id,
                                        ZXIC_UINT32 queue_id,
                                        ZXIC_UINT32 sdt_no);

DPP_STATUS dpp_dtb_hash_online_ddr_delete(ZXIC_UINT32 dev_id,
                                        ZXIC_UINT32 queue_id,
                                        ZXIC_UINT32 sdt_no,
                                        ZXIC_UINT32 flush_mode);

/** dtb dump 片内统计表项内容 支持64/128bit
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   dump_mode       dump类型 0-64bit 1-128bit
* @param   stat_cfg        ppu统计片内外基地址等配置
* @param   start_index     要dump的起始index，单位是dump_mode的64/128bit
* @param   entry_num       本次dump条目数（64/128bit为单位，最多支持2048/1024个）
* @param   p_dump_data_arr 本次dump出的数据，数据格式与下表格式相同（入参，用户配置）
* @return
* @remark  无
* @see
* @author  zth      @date  2024/06/17
************************************************************/
DPP_STATUS dpp_dtb_smmu0_stat_data_dump(ZXIC_UINT32 dev_id,
                               ZXIC_UINT32 queue_id,
                               ZXIC_UINT32 dump_mode,
                               PPU_STAT_CFG_T* stat_cfg,
                               ZXIC_UINT32 start_index,
                               ZXIC_UINT32 entry_num,
                               ZXIC_UINT8* p_dump_data_arr);

/** dtb dump 片外统计表项内容 支持64/128bit
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   dump_mode       dump类型 0-64bit 1-128bit
* @param   stat_cfg        ppu统计片内外基地址等配置
* @param   start_index     要dump的起始index，单位是dump_mode的64/128bit
* @param   entry_num       本次dump条目数（64/128bit为单位，最多支持2048/1024个）
* @param   p_dump_data_arr 本次dump出的数据，数据格式与下表格式相同（入参，用户配置）
* @return
* @remark  无
* @see
* @author  zth      @date  2024/06/17
************************************************************/
DPP_STATUS dpp_dtb_smmu1_stat_data_dump(ZXIC_UINT32 dev_id,
                                ZXIC_UINT32 queue_id,
                                ZXIC_UINT32 dump_mode,
                                PPU_STAT_CFG_T* stat_cfg,
                                ZXIC_UINT32 start_index,
                                ZXIC_UINT32 entry_num,
                                ZXIC_UINT8* p_dump_data_arr);

/***********************************************************/
/** 释放vport下的所有index
* @param   dev_id          NP设备号
* @param   sdt_no          流表sdt号(0~255)
* @param   vport           端口号
* @param   index           需要释放的索引值
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_acl_index_release_by_vport(ZXIC_UINT32 dev_id,
                                     ZXIC_UINT32 sdt_no,
                                     ZXIC_UINT32 vport);

/***********************************************************/
/** 释放当前sdt下的所有index
* @param   dev_id          NP设备号
* @param   sdt_no          流表sdt号(0~255)
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_acl_index_release_all(ZXIC_UINT32 dev_id,
                                     ZXIC_UINT32 sdt_no);

/***********************************************************/
/** 获取当前vport下分配的所有index
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   eram_sdt_no     维护index的eram直接表号
* @param   vport           端口号
* @param   index_num       出参，当前vport分配的index个数
* @param   p_index_array   出参，当前vport分配的index数组
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/18
************************************************************/
DPP_STATUS dpp_dtb_acl_index_parse(ZXIC_UINT32 dev_id,
                                ZXIC_UINT32 queue_id,
                                ZXIC_UINT32 eram_sdt_no,
                                ZXIC_UINT32 vport, 
                                ZXIC_UINT32 *index_num,
                                ZXIC_UINT32 *p_index_array);


/***********************************************************/
/** 清除指定index的所有eram表项
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   sdt_no          维护index的eram直接表号
* @param   index_num       当前vport分配的index个数
* @param   p_index_array   当前vport分配的index数组
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/18
************************************************************/
DPP_STATUS dpp_dtb_eram_data_clear(ZXIC_UINT32 dev_id, 
                                  ZXIC_UINT32 queue_id, 
                                  ZXIC_UINT32 sdt_no, 
                                  ZXIC_UINT32 index_num, 
                                  ZXIC_UINT32 *p_index_array);

/***********************************************************/
/** 清除指定index的所有acl表项
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   sdt_no          acl表项的sdt号
* @param   index_num       当前vport分配的index个数
* @param   p_index_array   当前vport分配的index数组
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/18
************************************************************/
DPP_STATUS dpp_dtb_acl_data_clear(ZXIC_UINT32 dev_id, 
                                  ZXIC_UINT32 queue_id, 
                                  ZXIC_UINT32 sdt_no, 
                                  ZXIC_UINT32 index_num, 
                                  ZXIC_UINT32 *p_index_array);

/***********************************************************/
/** 离线删除当前acl所有表项和索引值
* @param   dev_id          NP设备号
* @param   queue_id        dtb通道队列号(0~127)
* @param   sdt_no          流表sdt号(0~255)
* @param   vport           端口号
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_acl_offline_delete_all(ZXIC_UINT32 dev_id,
                                      ZXIC_UINT32 queue_id,
                                      ZXIC_UINT32 sdt_no);

/***********************************************************/
/** 获取指定handle列表的acl表项
* @param   dev_id          NP设备号
* @param   queue_id        dtb通道队列号(0~127)
* @param   sdt_no          流表sdt号(0~255)
* @param   index_num       handle个数
* @param   p_index_array   handle数组列表
* @param   pDumpData       出参,获取到的acl表项
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_acl_data_get_by_handle(ZXIC_UINT32 dev_id,
                                           ZXIC_UINT32 queue_id,
                                           ZXIC_UINT32 sdt_no,
                                           ZXIC_UINT32 index_num,
                                           ZXIC_UINT32* p_index_array,
                                           ZXIC_UINT8* pDumpData);

/***********************************************************/
/** 获取指定handle列表的统计值
* @param   dev_id          NP设备号
* @param   queue_id        dtb通道队列号(0~127)
* @param   p_stat_cfg      dump模式,起始偏移和深度
* @param   index_num       handle个数
* @param   p_index_array   handle数组列表
* @param   pDumpData       出参,获取到的acl表项
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_stat_data_get_by_handle(ZXIC_UINT32 dev_id,
                                           ZXIC_UINT32 queue_id,
                                           DPP_DTB_STAT_CFG_T *p_stat_cfg,
                                           ZXIC_UINT32 index_num,
                                           ZXIC_UINT32* p_index_array,
                                           ZXIC_UINT8* pDumpData);

/***********************************************************/
/** 指定vport的统计计数读清
* @param   dev_id           NP设备号
* @param   sdt_no           流表sdt号(0~255)
* @param   vport            端口号
* @param   rd_mode          读取位宽模式，参见STAT_CNT_MODE_E，0-64bit，1-128bit
* @param   start_counter_id 统计起始编号，对应微码中的address
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_acl_stat_cnt_clr(ZXIC_UINT32 dev_id,
                                     ZXIC_UINT32 sdt_no,
                                     ZXIC_UINT32 vport,
                                     STAT_CNT_MODE_E rd_mode,
                                     ZXIC_UINT32 start_counter_id);

/***********************************************************/
/** dtb方式清除指定index的所有统计项
* @param   dev_id          设备号
* @param   queue_id        队列号
* @param   counter_id      统计编号，对应微码中的address
* @param   rd_mode         统计读取方式 0:64bit 1:128bit
* @param   sdt_no          维护index的eram直接表号
* @param   index_num       当前vport分配的index个数
* @param   p_index_array   当前vport分配的index数组
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/18
************************************************************/
DPP_STATUS dpp_dtb_eram_stat_data_clear(ZXIC_UINT32 dev_id, 
                                  ZXIC_UINT32 queue_id, 
                                  ZXIC_UINT32 counter_id,
                                  STAT_CNT_MODE_E rd_mode,
                                  ZXIC_UINT32 index_num, 
                                  ZXIC_UINT32 *p_index_array);

/***********************************************************/
/** 统计计数读清(片外统计清除方式，发送消息通知riscv清除)
* @param   dev_id           NP设备号
* @param   rd_mode          读取位宽模式，参见STAT_CNT_MODE_E，0-64bit，1-128bit
* @param   start_count_id   统计起始编号，对应微码中的address
* @param   num              统计项个数
* @return
* @remark  无
* @see
* @author  cq      @date  2024/09/14
************************************************************/  
DPP_STATUS dpp_dtb_stat_ppu_smmu1_cnt_clr(ZXIC_UINT32 dev_id, 
                                     STAT_CNT_MODE_E rd_mode,
                                     ZXIC_UINT32 start_count_id,
                                     ZXIC_UINT32 num);
#endif


#ifdef __cplusplus
}
#endif

#endif
