
#ifndef _DPP_TLB_CFG_H_
#define _DPP_TLB_CFG_H_

#include "dpp_tlb_reg.h"

#ifdef __cplusplus
extern "C" {
#endif

#define TLB_WIDTH_MIN        (ZXIC_UINT32)(13)
#define TLB_VAWIDTH_RANGE0   (ZXIC_UINT32)(40)
#define TLB_PAWIDTH_RANGE0   (ZXIC_UINT32)(40)
#define TLB_PAWIDTH_RANGE1   (ZXIC_UINT32)(64)
#define TLB_PAWIDTH_RANGE2   (ZXIC_UINT32)(96)
#define TLB_ENTRY_NUM_MAX    (ZXIC_UINT32)(128)
#define TLB_ENTRY_NUM_MIN    (ZXIC_UINT32)(128)

typedef enum dpp_tlb_entry_num
{
    TLB_ENTRY_8    = 0,
    TLB_ENTRY_16   = 1,
    TLB_ENTRY_32   = 2,
    TLB_ENTRY_64   = 3,
    TLB_ENTRY_128  = 4,
    TLB_ENTRY_BUTT
}DPP_TLB_ENTRY_NUM;

typedef struct dpp_tlb_mgr_t
{
    ZXIC_UINT32 entry_num;     /**<  @brief 8/16/32/64/128*/
    ZXIC_UINT32 va_width;      /**<  @brief 最大位宽64bit*/
    ZXIC_UINT32 pa_width;      /**<  @brief 最大位宽128bit*/
}DPP_TLB_MGR_T;

typedef struct dpp_tlb_entry_cfg_t
{
    ZXIC_UINT8 valid;          /**<  @brief ENTRY有效控制 1:entry有效 0：entry无效 */
    ZXIC_UINT8 wr_valid;       /**<  @brief 写权限控制 1：有写权限 0：无写权限*/
    ZXIC_UINT8 rd_valid;       /**<  @brief 读权限控制 1：有读权限 0：无读权限*/
    ZXIC_UINT32 mask_l;        /**<  @brief VA_WIDTH<=40,mask[VA_WIDTH-1:12];VA_WIDTH>40,mask[39:12]*/
    ZXIC_UINT32 mask_h;        /**<  @brief VA_WIDTH>40,mask[VA_WIDTH-1:40]*/
    ZXIC_UINT32 va_l;          /**<  @brief VA_WIDTH<=40,虚拟基地址[VA_WIDTH-1:12];VA_WIDTH>40,虚拟基地址[39:12]*/
    ZXIC_UINT32 va_h;          /**<  @brief VA_WIDTH>40,虚拟基地址[VA_WIDTH-1:40]*/
    ZXIC_UINT32 pa_l;          /**<  @brief PA_WIDTH<=40,物理基地址[PA_WIDTH-1:12];PA_WIDTH>40,物理基地址[39:12]*/
    ZXIC_UINT32 pa_h0;         /**<  @brief PA_WIDTH<=64,物理基地址[PA_WIDTH-1:40];PA_WIDTH>64,物理基地址[63:40]*/
    ZXIC_UINT32 pa_h1;         /**<  @brief PA_WIDTH<=96,物理基地址[PA_WIDTH-1:64];PA_WIDTH>64,物理基地址[95:64]*/
    ZXIC_UINT32 pa_h2;         /**<  @brief PA_WIDTH>96,物理基地址[PA_WIDTH-1:96]*/
}DPP_TLB_ENTRY_CFG_T;

typedef struct dpp_tlb_pa_t
{
    ZXIC_UINT32 pa_l;
    ZXIC_UINT32 pa_h0;
    ZXIC_UINT32 pa_h1;
    ZXIC_UINT32 pa_h2;
}DPP_TLB_PA_T;

typedef struct dpp_tlb_va_mask_t
{
    ZXIC_UINT32 va_mask_l;
    ZXIC_UINT32 va_mask_h;
}DPP_TLB_VA_MASK_T;

/***********************************************************/
/** 创建TLB的管理结构
* @param   dev_id       设备号，支持多芯片  
*
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_mgr_create(ZXIC_UINT32 dev_id);

/***********************************************************/
/** 注销TLB的管理结构
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  zab      @date  2021/02/23
************************************************************/
ZXIC_UINT32 dpp_tlb_mgr_destory(ZXIC_UINT32 dev_id);

/***********************************************************/
/** 重置tlb管理结构
* @param   dev_id       设备号，支持多芯片   
*
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_mgr_reset(ZXIC_UINT32 dev_id);

/***********************************************************/
/** 获取tlb管理结构
* @param   dev_id               设备号，支持多芯片
*
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
DPP_TLB_MGR_T *dpp_tlb_mgr_get(ZXIC_UINT32 dev_id);

/***********************************************************/
/** tlb管理结构初始化
* @param   dev_id               设备号，支持多芯片
*
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_mgr_init(ZXIC_UINT32 dev_id);

/***********************************************************/
/** 初始化tlb基本配置信息
* @param   dev_id               设备号，支持多芯片
* @param   entry_num            实例数，DPP_TLB_ENTRY_NUM
* @param   va_width             虚拟地址位宽，最大位宽64bit
* @param   pa_width             物理地址位宽，最大位宽128bit
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_init(ZXIC_UINT32 dev_id,ZXIC_UINT32 entry_num,ZXIC_UINT32 va_width,ZXIC_UINT32 pa_width);

/***********************************************************/
/** 获取entry实例数
* @param   dev_id               设备号，支持多芯片
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_entry_num_get(ZXIC_UINT32 dev_id);

/***********************************************************/
/** 获取虚拟地址位宽
* @param   dev_id               设备号，支持多芯片
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_va_width_get(ZXIC_UINT32 dev_id);

/***********************************************************/
/** 获取物理地址位宽
* @param   dev_id               设备号，支持多芯片
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_pa_width_get(ZXIC_UINT32 dev_id);

/***********************************************************/
/** 获取tlb公共寄存器地址
* @param   dev_id               设备号，支持多芯片
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_global_reg_addr_get(ZXIC_UINT32 dev_id, ZXIC_UINT32 addr);

/***********************************************************/
/** 获取entry实例寄存器地址
* @param   dev_id               设备号，支持多芯片
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_entry_reg_addr_get(ZXIC_UINT32 dev_id, ZXIC_UINT32 addr, ZXIC_UINT32 entry_id);

/***********************************************************/
/** va地址或mask转换为寄存器配置格式
* @param   va_width   根据va位宽进行转换
* @param   pInVaMask  转换前的va或mask 
* @param   pOutVaMask 转换后的va或mask 
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
DPP_STATUS dpp_tlb_va_mask_write(ZXIC_UINT32 va_width,DPP_TLB_VA_MASK_T *pInVaMask,DPP_TLB_VA_MASK_T *pOutVaMask);

/***********************************************************/
/** 将寄存器读取出来的数据转换为正常数据
* @param   va_width   根据va位宽进行转换
* @param   pInVaMask  转换前的va或mask 
* @param   pOutVaMask 转换后的va或mask 
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
DPP_STATUS dpp_tlb_va_mask_read(ZXIC_UINT32 va_width,DPP_TLB_VA_MASK_T *pInVaMask,DPP_TLB_VA_MASK_T *pOutVaMask);

/***********************************************************/
/** pa地址转换为寄存器配置格式
* @param   pa_width   根据pa位宽进行转换
* @param   pInPa      转换前的pa 
* @param   pOutPa     转换后的pa 
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
DPP_STATUS dpp_tlb_pa_write(ZXIC_UINT32 pa_width,DPP_TLB_PA_T *pInPa,DPP_TLB_PA_T*pOutPa);

/***********************************************************/
/** 将寄存器读取出来的地址转换为正常pa地址
* @param   pa_width   根据pa位宽进行转换
* @param   pInPa      转换前的pa 
* @param   pOutPa     转换后的pa 
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
DPP_STATUS dpp_tlb_pa_read(ZXIC_UINT32 pa_width,DPP_TLB_PA_T *pInPa,DPP_TLB_PA_T*pOutPa);

/***********************************************************/
/** 配置TLB entry实例寄存器
* @param   dev_id          设备号，支持多芯片
* @param   entry_id        实例id 
* @param   pTlbEntryCfg    entry实例信息
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_entry_cfg_set(ZXIC_UINT32 dev_id,ZXIC_UINT32 entry_id, DPP_TLB_ENTRY_CFG_T *pTlbEntryCfg);

/***********************************************************/
/** 获取TLB entry实例寄存器信息
* @param   dev_id          设备号，支持多芯片
* @param   entry_id        实例id 
* @param   pTlbEntryCfg    出参，entry实例信息
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_entry_cfg_get(ZXIC_UINT32 dev_id,ZXIC_UINT32 entry_id, DPP_TLB_ENTRY_CFG_T *pTlbEntryCfg);

/***********************************************************/
/** 全局寄存器配置
* @param   dev_id          设备号，支持多芯片
* @param   pTlbGlobalCfg   全局寄存器信息
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_global_cfg_set(ZXIC_UINT32 dev_id, DPP_TLB_GLOBAL_CFG_T *pTlbGlobalCfg);

/***********************************************************/
/** 全局寄存器获取
* @param   dev_id          设备号，支持多芯片
* @param   pTlbGlobalCfg   全局寄存器信息
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_global_cfg_get(ZXIC_UINT32 dev_id, DPP_TLB_GLOBAL_CFG_T *pTlbGlobalCfg);

/***********************************************************/
/** 读错误发生后，继续工作模式下配置为输出固定地址时，输出此固定地址
* @param   dev_id          设备号，支持多芯片
* @param   addr   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_dsp0_output_addr_rd_set(ZXIC_UINT32 dev_id, ZXIC_UINT32 addr);

/***********************************************************/
/** 读错误发生后，继续工作模式下配置为输出固定地址时，输出此固定地址
* @param   dev_id          设备号，支持多芯片
* @param   pTlbOutputAddrRd   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_dsp0_output_addr_rd_get(ZXIC_UINT32 dev_id, DPP_TLB_DSP0_OUTPUT_ADDR_RD_T *pTlbOutputAddrRd);

/***********************************************************/
/** 写错误发生后，继续工作模式下配置为输出固定地址时，输出此固定地址
* @param   dev_id          设备号，支持多芯片
* @param   addr   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_dsp0_output_addr_rw_set(ZXIC_UINT32 dev_id, ZXIC_UINT32 addr);

/***********************************************************/
/** 写错误发生后，继续工作模式下配置为输出固定地址时，输出此固定地址
* @param   dev_id          设备号，支持多芯片
* @param   pTlbOutputAddrRw   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_dsp0_output_addr_rw_get(ZXIC_UINT32 dev_id, DPP_TLB_DSP0_OUTPUT_ADDR_RW_T *pTlbOutputAddrRw);

/***********************************************************/
/** 获取写地址错误状态和类型
* @param   dev_id          设备号，支持多芯片
* @param   pTlbRwErr   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_dsp0_rw_err_get(ZXIC_UINT32 dev_id, DPP_TLB_DSP0_RW_ERR_T *pTlbRwErr);

/***********************************************************/
/** 获取写地址错误地址
* @param   dev_id          设备号，支持多芯片
* @param   pTlbRwErrAddr   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_dsp0_rw_err_addr_get(ZXIC_UINT32 dev_id, DPP_TLB_DSP0_RW_ERR_ADDR_T *pTlbRwErrAddr);

/***********************************************************/
/** 获取写地址错误地址高位
* @param   dev_id          设备号，支持多芯片
* @param   pTlbRwErrAddrH   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_dsp0_rw_err_addr_high_get(ZXIC_UINT32 dev_id, DPP_TLB_DSP0_RW_ERR_ADDR_HIGH_T *pTlbRwErrAddrH);

/***********************************************************/
/** 获取读地址错误状态和类型
* @param   dev_id          设备号，支持多芯片
* @param   pTlbRdErr   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_dsp0_rd_err_get(ZXIC_UINT32 dev_id, DPP_TLB_DSP0_RD_ERR_T *pTlbRdErr);

/***********************************************************/
/** 获取读地址错误地址
* @param   dev_id          设备号，支持多芯片
* @param   pTlbRdErrAddr   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_dsp0_rd_err_addr_get(ZXIC_UINT32 dev_id, DPP_TLB_DSP0_RD_ERR_ADDR_T *pTlbRdErrAddr);

/***********************************************************/
/** 获取读地址错误地址高位
* @param   dev_id          设备号，支持多芯片
* @param   pTlbRdErrAddrH   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_dsp0_rd_err_addr_high_get(ZXIC_UINT32 dev_id, DPP_TLB_DSP0_RD_ERR_ADDR_HIGH_T *pTlbRdErrAddrH);

/***********************************************************/
/** 清除寄存器，用APB总线直接进行写，写1表示清除
* @param   dev_id          设备号，支持多芯片
* @param   pTlbIntClr   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_o_dsp0_int_clr_set(ZXIC_UINT32 dev_id, ZXIC_UINT32 dsp0_int_clr);

/***********************************************************/
/**  掩码输入信号，0：屏蔽 1：不屏蔽 默认不屏蔽
* @param   dev_id          设备号，支持多芯片
* @param   int_enable_in   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_int_enable_in_set(ZXIC_UINT32 dev_id, ZXIC_UINT32 int_enable_in);

/***********************************************************/
/**  掩码输入信号，0：屏蔽 1：不屏蔽 默认不屏蔽
* @param   dev_id          设备号，支持多芯片
* @param   pTlbIntEnaIn   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_int_enable_in_get(ZXIC_UINT32 dev_id, DPP_TLB_INT_ENABLE_IN_T *pTlbIntEnaIn);

/***********************************************************/
/** 表示清除清除状态输入信号，软件配置写1表示清除
* @param   dev_id          设备号，支持多芯片
* @param   pTlbIntClrIn   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_int_clr_in_set(ZXIC_UINT32 dev_id, ZXIC_UINT32 int_clr_in);

/***********************************************************/
/** 中断注入选择输入，写1表示软件模拟外部产生中断
* @param   dev_id          设备号，支持多芯片
* @param   pTlbIntSetIn   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_int_set_in_set(ZXIC_UINT32 dev_id, ZXIC_UINT32 int_set_in);

/***********************************************************/
/** 中断状态输出信号，读出1表示此位有中断输出
* @param   dev_id          设备号，支持多芯片
* @param   int_sta_out   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_int_sta_out_set(ZXIC_UINT32 dev_id,ZXIC_UINT32 int_sta_out);

/***********************************************************/
/** TLB写/读的VA_PA标志，1表示VA_PA_SWITO_IN/VA_PA_SWIT1_IN完成
* @param   dev_id          设备号，支持多芯片
* @param   pTlbVaPaVaSwitFlag   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_va_pa_va_swit_flag_get(ZXIC_UINT32 dev_id, DPP_TLB_VA_PA_VA_SWIT_FLAG_T *pTlbVaPaVaSwitFlag);

/***********************************************************/
/** 写地址错误的ID，锁存首次错误ID
* @param   dev_id          设备号，支持多芯片
* @param   pTlbVaPaVaSwitFlag   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_write_err_aid_get(ZXIC_UINT32 dev_id, DPP_TLB_WRITE_ERR_AID_T *pTlbWriteErrAid);

/***********************************************************/
/** 读地址错误的ID，锁存首次错误ID
* @param   dev_id          设备号，支持多芯片
* @param   pTlbVaPaVaSwitFlag   
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2024/01/23
************************************************************/
ZXIC_UINT32 dpp_tlb_read_err_aid_get(ZXIC_UINT32 dev_id, DPP_TLB_READ_ERR_AID_T *pTlbReadErrAid);

/***********************************************************/
/** 物理地址配置1
* @param   dev_id          设备号，支持多芯片
* @param   pa_h2            物理地址高32bit [127:96]
* @param   pa_h1            物理地址高32bit [95:64]
* @param   pa_h0            物理地址高32bit [63:32]
* @param   pa_l             物理地址低32bit [31:0]
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_pa_va_swit0_set(ZXIC_UINT32 dev_id, ZXIC_UINT32 pa_h2,ZXIC_UINT32 pa_h1,ZXIC_UINT32 pa_h0,ZXIC_UINT32 pa_l);

/***********************************************************/
/** 物理地址配置2
* @param   dev_id          设备号，支持多芯片
* @param   pa_h2            物理地址高32bit [127:96]
* @param   pa_h1            物理地址高32bit [95:64]
* @param   pa_h0            物理地址高32bit [63:32]
* @param   pa_l             物理地址低32bit [31:0]
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_pa_va_swit1_set(ZXIC_UINT32 dev_id, ZXIC_UINT32 pa_h2,ZXIC_UINT32 pa_h1,ZXIC_UINT32 pa_h0,ZXIC_UINT32 pa_l);

/***********************************************************/
/** 读取配置的寄存器信息1
* @param   dev_id          设备号，支持多芯片
* @param   pa_l             物理地址低32bit [31:0]
* @param   pa_h0            物理地址高32bit [63:32]
* @param   pa_h1            物理地址高32bit [95:64]
* @param   pa_h2            物理地址高32bit [127:96]
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_pa_va_swit0_get(ZXIC_UINT32 dev_id, DPP_TLB_PA_T *pTlbPa);

/***********************************************************/
/** 读取配置的寄存器信息2
* @param   dev_id          设备号，支持多芯片
* @param   pa_l             物理地址低32bit [31:0]
* @param   pa_h0            物理地址高32bit [63:32]
* @param   pa_h1            物理地址高32bit [95:64]
* @param   pa_h2            物理地址高32bit [127:96]
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_pa_va_swit1_get(ZXIC_UINT32 dev_id, DPP_TLB_PA_T *pTlbPa);

/***********************************************************/
/** 读取转换完后的物理地址信息1
* @param   dev_id          设备号，支持多芯片
* @param   pTlbPa          出参，物理地址信息
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_va_pa_swit0_out_get(ZXIC_UINT32 dev_id, DPP_TLB_PA_T *pTlbPa);

/***********************************************************/
/** 读取转换完后的物理地址信息2
* @param   dev_id          设备号，支持多芯片
* @param   pTlbPa          出参，物理地址信息
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_va_pa_swit1_out_get(ZXIC_UINT32 dev_id, DPP_TLB_PA_T *pTlbPa);

/***********************************************************/
/** 虚拟地址配置1
* @param   dev_id          设备号，支持多芯片
* @param   va_h            虚拟地址高32bit [63:32]
* @param   va_l            虚拟地址低32bit [31:0]
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_va_pa_swit0_set(ZXIC_UINT32 dev_id, ZXIC_UINT32 va_h,ZXIC_UINT32 va_l);

/***********************************************************/
/** 虚拟地址配置2
* @param   dev_id          设备号，支持多芯片
* @param   va_h            虚拟地址高32bit [63:32]
* @param   va_l            虚拟地址低32bit [31:0]
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_va_pa_swit0_set(ZXIC_UINT32 dev_id, ZXIC_UINT32 va_h,ZXIC_UINT32 va_l);

/***********************************************************/
/** 虚拟地址读取1
* @param   dev_id          设备号，支持多芯片
* @param   pTlbVa          虚拟地址指针
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_va_pa_swit0_get(ZXIC_UINT32 dev_id, DPP_TLB_VA_MASK_T *pTlbVa);

/***********************************************************/
/** 虚拟地址读取2
* @param   dev_id          设备号，支持多芯片
* @param   pTlbVa          虚拟地址指针
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_va_pa_swit1_get(ZXIC_UINT32 dev_id, DPP_TLB_VA_MASK_T *pTlbVa);

/***********************************************************/
/** 读取转换后的虚拟地址1
* @param   dev_id          设备号，支持多芯片
* @param   pTlbVa          出参，虚拟地址指针
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_pa_va_swit0_out_get(ZXIC_UINT32 dev_id, DPP_TLB_VA_MASK_T *pTlbVa);

/***********************************************************/
/** 读取转换后的虚拟地址2
* @param   dev_id          设备号，支持多芯片
* @param   pTlbVa          出参，虚拟地址指针
* @return  
* @remark  无
* @see     
* @author  chenqin00181032      @date  2023/04/18
************************************************************/
ZXIC_UINT32 dpp_tlb_pa_va_swit1_out_get(ZXIC_UINT32 dev_id, DPP_TLB_VA_MASK_T *pTlbVa);

#ifdef __cplusplus
}
#endif

#endif